with Ada.Characters.Latin_1; use Ada.Characters.Latin_1;
with Ada.Strings.UTF_Encoding.Wide_Wide_Strings;
use Ada.Strings.UTF_Encoding.Wide_Wide_Strings;

with Win32.Winbase;

package body Windows_Support is

   use System;
   use Win32;
   use type Win32.DWORD;

   -----------------------
   -- Sys_Error_Message --
   -----------------------

   function Sys_Error_Message (Error_Code : Win32.DWORD) return UTF_8_String is
      Buffer_Address : aliased System.Address;
      Reached_Finally : Boolean := False;
      procedure Finally is
      begin
         if Reached_Finally then
            return;
         end if;
         Reached_Finally := True;

         -- Free the OS allocated memory block
         Buffer_Address := Winbase.LocalFree (Buffer_Address);
         if Buffer_Address /= Null_Address then
            Raise_Last_OS_Error;
         end if;
      end Finally;
      Len : Win32.DWORD;
   begin
      -- Obtain the formatted message for the given Win32 ErrorCode
      -- Let the OS initialize the Buffer variable. Need to LocalFree it
      -- afterward.
      Len := Winbase.FormatMessageW
        (Winbase.FORMAT_MESSAGE_FROM_SYSTEM or
           Winbase.FORMAT_MESSAGE_IGNORE_INSERTS or
             Winbase.FORMAT_MESSAGE_ALLOCATE_BUFFER,
         Null_Address, Error_Code, 0,
         Win32.LPWSTR (WCHAR_A2A.To_Pointer (Buffer_Address'Address)), 0);

      declare
         Buffer : UTF_16_Wide_String (1 .. Natural (Len))
           with Import, Convention => C, Address => Buffer_Address;
      begin
         -- Remove the undesired line breaks and '.' char
         while Len > 0 and then
           Buffer (Natural (Len)) in Wide_Character'Val (0) .. ' ' | '.'
         loop
            Len := Len - 1;
         end loop;

         -- Convert to Ada string
         return Result : UTF_8_String :=
           Encode (Decode (Buffer (1 .. Natural (Len))))
         do
            Finally;
         end return;
      exception
         when others => Finally; raise;
      end;
   end Sys_Error_Message;

   -------------------------
   -- Raise_Last_OS_Error --
   -------------------------

   procedure Raise_Last_OS_Error is
   begin
      Raise_Last_OS_Error (Winbase.GetLastError);
   end;

   -------------------------
   -- Raise_Last_OS_Error --
   -------------------------

   procedure Raise_Last_OS_Error (Last_Error : Win32.DWORD) is
   begin
      if Last_Error /= 0 then
         raise Windows_Error with "System Error. Code:" &
           Win32.DWORD'Image (Last_Error) & '.' & LF &
           Sys_Error_Message (Last_Error);
      else
         raise Windows_Error with "A call to an OS function failed";
      end if;
   end;

   --------------------
   -- Check_OS_Error --
   --------------------

   procedure Check_OS_Error (Last_Error : Win32.DWORD) is
   begin
      if Last_Error /= 0 then
         Raise_Last_OS_Error (Last_Error);
      end if;
   end Check_OS_Error;

   --------------------
   -- Check_OS_Error --
   --------------------

   procedure Check_OS_Error (Ret_Val : Win32.BOOL) is
   begin
      if 0 = Ret_Val then
         Raise_Last_OS_Error;
      end if;
   end Check_OS_Error;

end Windows_Support;
