unit WaveIn;

interface

uses
  SysUtils, Windows, MMSystem;

type
  EWaveInError = class(Exception);

type
  TWaveInOnRead = procedure(const Buf: TWaveHdr) of object;
  TWaveInState = (wsStopped, wsRecording);

type
  TWaveIn = class(TObject)
    private
      FHandle: HWAVEIN;
      FState: TWaveInState;
      FBuf: array[0..1] of TWaveHdr;
      FBufInd: Integer;
      procedure AllocBuf(var Buf: TWaveHdr);
      procedure FreeBuf(var Buf: TWaveHdr);
      procedure Open;
      procedure Close;
      procedure Read;
      procedure AddBuffer;
    public
      DeviceID: Integer;
      OnRead: TWaveInOnRead;
      BufferLength: Integer;
      Format: TWaveFormatEx;
      property Handle: HWAVEIN read FHandle;
      property State: TWaveInState read FState;
      constructor Create(ADeviceID: Integer; const AFormat: TWaveFormatEx;
        AOnRead: TWaveInOnRead);
      constructor CreatePCM(ADeviceID, PCMRate, PCMBits, PCMChannels: Integer;
        AOnRead: TWaveInOnRead);
      destructor Destroy; override;
      procedure Start;
      procedure Stop;
      procedure GetPosition(var Pos: TMMTIME);
  end;

implementation

procedure WaveInProc(hwo: HWAVEIN; uMsg: UINT; dwInstance,
  dwParam1, dwParam2: DWord); stdcall;
begin
  with TWaveIn(dwInstance) do
    if (uMsg = MM_WIM_DATA) and (FState = wsRecording) then Read;

{    if uMsg = MM_WIM_DATA then
      if FState = wsStopped then
        FreeBuf(PWaveHdr(dwParam1)^)
      else
        Read;}

end;

constructor TWaveIn.Create(ADeviceID: Integer;
  const AFormat: TWaveFormatEx; AOnRead: TWaveInOnRead);
begin
  DeviceID := ADeviceID;
  OnRead := AOnRead;
  Format := AFormat;
  FState := wsStopped;
  BufferLength := AFormat.nSamplesPerSec * 2;
end;

constructor TWaveIn.CreatePCM(ADeviceID: Integer;
  PCMRate, PCMBits, PCMChannels: Integer; AOnRead: TWaveInOnRead);
var
  wf: TWaveFormatEx;
begin
  wf.cbSize := SizeOf(TWaveFormatEx);
  wf.wFormatTag := WAVE_FORMAT_PCM;
  wf.nSamplesPerSec := PCMRate;
  wf.wBitsPerSample := PCMBits;
  wf.nChannels := PCMChannels;
  wf.nBlockAlign := (PCMBits shr 3) * PCMChannels;
  wf.nAvgBytesPerSec := wf.nBlockAlign * PCMRate;
  Create(ADeviceID, wf, AOnRead);
end;

destructor TWaveIn.Destroy;
begin
  Stop;
end;

procedure TWaveIn.AllocBuf(var Buf: TWaveHdr);
begin
  Buf.dwBufferLength := Format.nBlockAlign * BufferLength;
  GetMem(Buf.lpData, Buf.dwBufferLength);
  FillChar(Buf.lpData^, Buf.dwBufferLength, 0);
  Buf.dwFlags := 0;
  if waveInPrepareHeader(FHandle, @Buf,
    SizeOf(TWAVEHDR)) <> MMSYSERR_NOERROR then
    raise EWaveInError.Create('Unable to initialize input buffer');
end;

procedure TWaveIn.FreeBuf(var Buf: TWaveHdr);
begin
  if Buf.dwFlags and WHDR_PREPARED <> 0 then
    waveInUnprepareHeader(FHandle, @Buf, SizeOf(TWAVEHDR));
  FreeMem(Buf.lpData);
end;

procedure TWaveIn.Open;
begin
  if waveInOpen(@FHandle, DeviceID, @Format, Integer(@WaveInProc),
    Integer(Self), CALLBACK_FUNCTION) <> MMSYSERR_NOERROR then
    begin
      FHandle := 0;
      raise EWaveInError.Create('Unable to open waveform-input device');
    end;
  AllocBuf(FBuf[0]);
  AllocBuf(FBuf[1]);
  FBufInd := 0;
end;

procedure TWaveIn.Close;
begin
  if FHandle <> 0 then
    begin
      waveInReset(FHandle);
      FreeBuf(FBuf[1]);
      FreeBuf(FBuf[0]);
      waveInClose(FHandle);
      FHandle := 0;
    end;
end;

procedure TWaveIn.Read;
begin
  if Assigned(OnRead) then OnRead(FBuf[FBufInd]);
  AddBuffer;
end;

procedure TWaveIn.Start;
begin
  if State = wsRecording then Exit;
  Open;
  AddBuffer;
  AddBuffer;
  if waveInStart(FHandle) <> MMSYSERR_NOERROR then
    raise EWaveInError.Create('Unable to start recording om waveform-input device');
  FState := wsRecording;
end;

procedure TWaveIn.Stop;
begin
  if State = wsStopped then Exit;
  FState := wsStopped;
  Close;
end;

procedure TWaveIn.AddBuffer;
begin
  if waveInAddBuffer(FHandle, @FBuf[FBufInd], SizeOf(TWaveHdr)) <> MMSYSERR_NOERROR then
    raise EWaveInError.Create('Unable to add input buffer');
  FBufInd := FBufInd xor 1;
end;

procedure TWaveIn.GetPosition(var Pos: TMMTIME);
begin
  if waveInGetPosition(FHandle, @Pos, SizeOf(TMMTIME)) <> MMSYSERR_NOERROR then
    raise EWaveInError.Create('Unable to get playback position');
end;

end.
