unit WaveOut;

interface

uses
  SysUtils, Windows, MMSystem;

type
  EWaveOutError = class(Exception);

type
  TWaveOutOnRender = procedure(var Buf: TWaveHdr) of object;
  TWaveOutState = (wsStopped, wsPlaying, wsPaused);

type
  TWaveOut = class(TObject)
    private
      FHandle: HWAVEOUT;
      FState: TWaveOutState;
      FBuf: array[0..1] of TWaveHdr;
      FBufInd: Integer;
      procedure AllocBuf(var Buf: TWaveHdr);
      procedure FreeBuf(var Buf: TWaveHdr);
      procedure Open;
      procedure Close;
      procedure Write;
      procedure Done;
    public
      DeviceID: Integer;
      OnRender: TWaveOutOnRender;
      BufferLength: Integer;
      Format: TWaveFormatEx;
      property Handle: HWAVEOUT read FHandle;
      property State: TWaveOutState read FState;
      constructor Create(ADeviceID: Integer; const AFormat: TWaveFormatEx;
        AOnRender: TWaveOutOnRender);
      constructor CreatePCM(ADeviceID: Integer;
        PCMRate, PCMBits, PCMChannels: Integer; AOnRender: TWaveOutOnRender);
      destructor Destroy; override;
      procedure Play;
      procedure Pause;
      procedure Stop;
      procedure GetPosition(var Pos: TMMTIME);
  end;

implementation

procedure WaveOutProc(hwo: HWAVEOUT; uMsg: UINT; dwInstance,
  dwParam1, dwParam2: DWord); stdcall;
begin
  with TWaveOut(dwInstance) do
    if (uMsg = MM_WOM_DONE) and (FState = wsPlaying) then Done;
end;

constructor TWaveOut.Create(ADeviceID: Integer; const AFormat: TWaveFormatEx; AOnRender: TWaveOutOnRender);
begin
  DeviceID := ADeviceID;
  OnRender := AOnRender;
  Format := AFormat;
  FState := wsStopped;
  BufferLength := AFormat.nSamplesPerSec * 2;
end;

constructor TWaveOut.CreatePCM(ADeviceID: Integer;
  PCMRate, PCMBits, PCMChannels: Integer; AOnRender: TWaveOutOnRender);
var
  wf: TWaveFormatEx;
begin
  wf.cbSize := SizeOf(TWaveFormatEx);
  wf.wFormatTag := WAVE_FORMAT_PCM;
  wf.nSamplesPerSec := PCMRate;
  wf.wBitsPerSample := PCMBits;
  wf.nChannels := PCMChannels;
  wf.nBlockAlign := (PCMBits shr 3) * PCMChannels;
  wf.nAvgBytesPerSec := wf.nBlockAlign * PCMRate;
  Create(ADeviceID, wf, AOnRender);
end;

destructor TWaveOut.Destroy;
begin
  Stop;
end;

procedure TWaveOut.AllocBuf(var Buf: TWaveHdr);
begin
  Buf.dwBufferLength := Format.nBlockAlign * BufferLength;
  GetMem(Buf.lpData, Buf.dwBufferLength);
  FillChar(Buf.lpData^, Buf.dwBufferLength, 0);
  Buf.dwFlags := 0;
  if waveOutPrepareHeader(FHandle, @Buf,
    SizeOf(TWAVEHDR)) <> MMSYSERR_NOERROR then
    raise EWaveOutError.Create('Unable to initialize output buffer');
end;

procedure TWaveOut.FreeBuf(var Buf: TWaveHdr);
begin
  if Buf.dwFlags and WHDR_PREPARED <> 0 then
    waveOutUnprepareHeader(FHandle, @Buf, SizeOf(TWAVEHDR));
  FreeMem(Buf.lpData);
end;

procedure TWaveOut.Open;
begin
  if waveOutOpen(@FHandle, DeviceID, @Format, Integer(@WaveOutProc),
    Integer(Self), CALLBACK_FUNCTION) <> MMSYSERR_NOERROR then
    begin
      FHandle := 0;
      raise EWaveOutError.Create('Unable to open waveform-output device');
    end;
  AllocBuf(FBuf[0]);
  AllocBuf(FBuf[1]);
  FBufInd := 0;
end;

procedure TWaveOut.Close;
begin
  if FHandle <> 0 then
    begin
      waveOutReset(FHandle);
      FreeBuf(FBuf[1]);
      FreeBuf(FBuf[0]);
      waveOutClose(FHandle);
      FHandle := 0;
    end;
end;

procedure TWaveOut.Write;
begin
  if waveOutWrite(FHandle, @FBuf[FBufInd],
    SizeOf(TWAVEHDR)) <> MMSYSERR_NOERROR then
    raise EWaveOutError.Create('Unable to write output buffer');
  FBufInd := FBufInd xor 1;
end;

procedure TWaveOut.Done;
begin
  if Assigned(OnRender) then
    begin
      OnRender(FBuf[FBufInd]);
      Write;
    end;
end;

procedure TWaveOut.Play;
begin
  if State = wsPlaying then Exit;
  if State = wsStopped then
    begin
      Open;
      Done;
      Done;
    end
  else
    waveOutRestart(FHandle);
  FState := wsPlaying;
end;

procedure TWaveOut.Stop;
begin
  if State = wsStopped then Exit;
  FState := wsStopped;
  Close;
end;

procedure TWaveOut.Pause;
begin
  if State <> wsPlaying then Exit;
  waveOutPause(FHandle);
  FState := wsPaused;
end;

procedure TWaveOut.GetPosition(var Pos: TMMTIME);
begin
  if waveOutGetPosition(FHandle, @Pos, SizeOf(TMMTIME)) <> MMSYSERR_NOERROR then
    raise EWaveOutError.Create('Unable to get playback position');
end;

end.
