unit ASG;

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Graphics, Grids, Clipbrd;

type
  TFormulaEvent = procedure(ACol, ARow: Integer; var Data: String) of object;

type
  TAdvancedStringGrid = class;

  TCellAttributes = class(TObject)
    private
      FGrid: TAdvancedStringGrid;
      FFill: TBrush;
      FFont: TFont;
      FAlignment: TAlignment;
      procedure SetFill(Value: TBrush);
      procedure SetFont(Value: TFont);
      procedure SetAlignment(Value: TAlignment);
      procedure FontChange(Sender: TObject);
      procedure BrushChange(Sender: TObject);
    public
      Data: TObject;
      constructor Create(AGrid: TAdvancedStringGrid);
      destructor Destroy; override;
      procedure Assign(Value: TCellAttributes);
      property Fill: TBrush read FFill write SetFill;
      property Font: TFont read FFont write SetFont;
      property Alignment: TAlignment read FAlignment write SetAlignment;
    end;

  TAdvancedStringGrid = class(TStringGrid)
    private
      FWordWrap: Boolean;
      FOnFormula: TFormulaEvent;
      function EnsureAttrib(ACol, ARow: Integer): TCellAttributes;
      function GetAttributes(ACol, ARow: Integer): TCellAttributes;
      procedure SetAttributes(ACol, ARow: Integer; Value: TCellAttributes);
      function GetObject(ACol, ARow: Integer): TObject;
      procedure SetObject(ACol, ARow: Integer; Value: TObject);
      procedure SetWordWrap(Value: Boolean);
      procedure AlignAttachedControls;
    protected
      procedure DrawCell(ACol, ARow: Longint;
        ARect: TRect; AState: TGridDrawState); override;
      procedure KeyDown(var Key: Word; Shift: TShiftState); override;
      procedure ColWidthsChanged; override;
      procedure RowHeightsChanged; override;
    public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure DeleteColumn(ACol: LongInt); override;
      procedure DeleteRow(ARow: LongInt); override;
      procedure CopyToClipboard(Cut: Boolean); virtual;
      procedure PasteFromClipboard; virtual;
//      procedure InsertControl;
      property Objects[ACol, ARow: Integer]: TObject
        read GetObject write SetObject;
      property Attributes[ACol, ARow: Integer]: TCellAttributes
        read GetAttributes write SetAttributes;
      property WordWrap: Boolean read FWordWrap write SetWordWrap;
      property OnFormula: TFormulaEvent read FOnFormula write FOnFormula;
  end;

procedure Register;

implementation

{ TCellAttributes }

procedure TCellAttributes.SetFill(Value: TBrush);
begin
  FFill.Assign(Value);
end;

procedure TCellAttributes.SetFont(Value: TFont);
begin
  FFont.Assign(Value);
end;

procedure TCellAttributes.SetAlignment(Value: TAlignment);
begin
  if Value = FAlignment then Exit;
  FAlignment := Value;
  FGrid.Invalidate;
end;

procedure TCellAttributes.FontChange(Sender: TObject);
begin
  FGrid.Invalidate;
end;

procedure TCellAttributes.BrushChange(Sender: TObject);
begin
  FGrid.Invalidate;
end;

procedure TCellAttributes.Assign(Value: TCellAttributes);
begin
  if Value = Self then Exit;
  FFill.Assign(Value.Fill);
  FFont.Assign(Value.Font);
  FGrid.Repaint;
end;

constructor TCellAttributes.Create(AGrid: TAdvancedStringGrid);
begin
  inherited Create;
  FGrid := AGrid;
  FFill := TBrush.Create;
  FFill.OnChange := BrushChange;
  FFill.Style := bsClear;
  FFont := TFont.Create;
  FFont.OnChange := FontChange;
  FAlignment := taLeftJustify;
end;

destructor TCellAttributes.Destroy;
begin
  FFont.Free;
  FFill.Free;
  inherited Destroy;
end;

{ TAdvancedStringGrid }

function TAdvancedStringGrid.GetObject(ACol, ARow: Integer): TObject;
begin
  Result := Attributes[ACol, ARow].Data;
end;

procedure TAdvancedStringGrid.SetObject(ACol, ARow: Integer; Value: TObject);
begin
  Attributes[ACol, ARow].Data := Value;
end;

function TAdvancedStringGrid.EnsureAttrib(ACol, ARow: Integer): TCellAttributes;
begin
  if inherited Objects[ACol, ARow] = nil then
    inherited Objects[ACol, ARow] := TCellAttributes.Create(Self);
  Result := TCellAttributes(inherited Objects[ACol, ARow]);
end;

function TAdvancedStringGrid.GetAttributes(ACol, ARow: Integer): TCellAttributes;
begin
  Result := EnsureAttrib(ACol, ARow);
end;

procedure TAdvancedStringGrid.SetAttributes(ACol, ARow: Integer;
  Value: TCellAttributes);
begin
  EnsureAttrib(ACol, ARow).Assign(Value);
end;

function AlignOfs(PageWidth, TextWidth: Integer;
  Align: TAlignment): Integer;
begin
  if Align = taCenter then
    Result := (PageWidth - TextWidth) div 2
  else
    if Align = taRightJustify then
      Result := PageWidth - TextWidth
    else
      Result := 0;
  if Result < 0 then Result := 0;
end;

const
  TextMargin = 2;

procedure TAdvancedStringGrid.DrawCell(ACol, ARow: Longint; ARect: TRect;
  AState: TGridDrawState);
var
  cw, y, i, s,
  LineWidth, SpWidth: Integer;
  Text, Word, Line: String;
  OutRect: TRect;

begin
  Text := Cells[ACol, ARow];
  if Assigned(FOnFormula) and (Text <> '') and (Text[1] = '=') then
    FOnFormula(ACol, ARow, Text);
  EnsureAttrib(ACol, ARow);
  OutRect := Rect(ARect.Left + TextMargin, ARect.Top + TextMargin,
    ARect.Right - TextMargin, ARect.Bottom - TextMargin);
  cw := OutRect.Right - OutRect.Left;
  with Canvas do
    begin
//      if AState = [] then
      Canvas.Brush.Assign(Attributes[ACol, ARow].Fill);
      Font.Assign(Attributes[ACol, ARow].Font);
      if gdSelected in AState then
        begin
          Brush.Color := clHighlight;
          Brush.Style := bsSolid;
          Font.Color := clWhite;
        end;
      Pen.Style := psSolid;
      Pen.Width := 1;
      Pen.Color := 0;
{      MoveTo(ARect.Left, ARect.Bottom);
      LineTo(ARect.Left, ARect.Top);
      LineTo(ARect.Right, ARect.Top);
      Pen.Style := psClear;}
      Rectangle(ARect.Left, ARect.Top, ARect.Right - 1, ARect.Bottom - 1); 
      Brush.Style := bsClear;
      if FWordWrap then
        begin
          SpWidth := TextWidth(' ');
          Word := '';
          LineWidth := 0;
          y := 0;
          Line := '';
          Text := Text + ' ';
          for i := 1 to Length(Text) do
            if Text[i] > ' ' then
              Word := Word + Text[i]
            else
              if Word <> '' then
                begin
                  s := TextWidth(Word);
                  if LineWidth + s + SpWidth > cw then
                    begin
                      TextRect(OutRect, OutRect.Left + AlignOfs(cw, LineWidth,
                        Attributes[ACol, ARow].Alignment), OutRect.Top + y, Line);
                      Inc(y, TextHeight(Line));
                      Line := Word;
                      LineWidth := s;
                    end
                  else
                    begin
                      if Line <> '' then
                        begin
                          Line := Line + ' ';
                          Inc(LineWidth, SpWidth);
                        end;
                      Line := Line + Word;
                      Inc(LineWidth, s);
                    end;
                  Word := '';
                end;
//      if Line <> '' then
          TextRect(OutRect, OutRect.Left + AlignOfs(cw, LineWidth,
            Attributes[ACol, ARow].Alignment), OutRect.Top + y, Line);
        end // if FWordWrap
      else
        TextRect(OutRect, OutRect.Left + AlignOfs(cw, TextWidth(Text),
            Attributes[ACol, ARow].Alignment), OutRect.Top
              + (OutRect.Bottom - OutRect.Top - TextHeight(Text)) shr 1, Text);
    end; // with Canvas
end;

procedure TAdvancedStringGrid.SetWordWrap(Value: Boolean);
begin
  if Value = FWordWrap then Exit;
  FWordWrap := Value;
  Invalidate;
end;

procedure TAdvancedStringGrid.DeleteColumn(ACol: LongInt);
var
  i: Integer;
begin
  for i := 0 to RowCount - 1 do
    if inherited Objects[ACol, i] <> nil then
      TCellAttributes(inherited Objects[ACol, i]).Free;
end;

procedure TAdvancedStringGrid.DeleteRow(ARow: LongInt);
var
  i: Integer;
begin
  for i := 0 to ColCount - 1 do
    if inherited Objects[i, ARow] <> nil then
      TCellAttributes(inherited Objects[i, ARow]).Free;
end;

constructor TAdvancedStringGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FWordWrap := False;
  FOnFormula := nil;
end;

destructor TAdvancedStringGrid.Destroy;
var
  i, j: Integer;
begin
  for i := 0 to RowCount - 1 do
    for j := 0 to ColCount - 1 do
      if inherited Objects[i, j] <> nil then
        TCellAttributes(inherited Objects[i, j]).Free;
  inherited Destroy;
end;

procedure TAdvancedStringGrid.CopyToClipboard(Cut: Boolean);
var
  i, j: Integer;
  s: String;
begin
  s := '';
  for i := Selection.Top to Selection.Bottom do
    begin
      for j := Selection.Left to Selection.Right do
        begin
          s := s + Cells[j, i];
          if j <> Selection.Right then s := s + #9;
          if Cut then Cells[j, i] := '';
        end;
      s := s + #13#10;
    end;
  if s <> '' then
    with TClipboard.Create do
      begin
        Open;
        AsText := s;
        Close;
        Free;
      end;
end;

procedure TAdvancedStringGrid.PasteFromClipboard;
var
  s, Data: String;
  i, Col, Row: Integer;
  Cr: Boolean;
begin
  with TClipboard.Create do
    begin
      Col := 0;
      Row := 0;
      s := AsText;
      Data := '';
      Cr := False;
      for i := 1 to Length(s) do
        begin
          case s[i] of
            #9:
              begin
                Cells[Selection.Left + Col, Selection.Top + Row] := Data;
                Data := '';
                Inc(Col);
              end;
            #10:
              if Cr then
                begin
                  Cells[Selection.Left + Col, Selection.Top + Row] := Data;
                  Data := '';
                  Col := 0;
                  Inc(Row);
                end
              else
                Data := Data + #13;
              #13:
                Cr := True;
            else
              Data := Data + s[i];
          end; // case s[i] of
          Cr := Cr and (s[i] = #13);
        end;
      Free;
    end;
end;

procedure TAdvancedStringGrid.KeyDown(var Key: Word; Shift: TShiftState);
begin
   if Shift = [ssCtrl] then
     case Chr(Key) of
       'C': CopyToClipboard(False);
       'X': CopyToClipboard(True);
       'V': PasteFromClipboard;
       #45: CopyToClipboard(False);
     end
   else
     if Shift = [ssShift] then
       case Key of
         45: PasteFromClipboard;
         46: CopyToClipboard(True);
       end;
  inherited KeyDown(Key, Shift);
end;

procedure TAdvancedStringGrid.AlignAttachedControls;
var
  i: Integer;
begin
  for i := 0 to ControlCount - 1 do
    begin
    end;
end;

procedure TAdvancedStringGrid.ColWidthsChanged;
begin
  inherited ColWidthsChanged;
  AlignAttachedControls;
end;

procedure TAdvancedStringGrid.RowHeightsChanged;
begin
  inherited RowHeightsChanged;
  AlignAttachedControls;
end;

procedure Register;
begin
  RegisterComponents('Samples', [TAdvancedStringGrid]);
end;

end.
