unit SnapinData;

interface

uses windows, classes, sysutils, controls, ComObj, menus, mmc_tlb, unitVersionInfo;

type
TScopeItem = class;
TScopeItems = class;
TViewType = (vtListView, vtGUID, vtHTML);
TSnapinColumns = class;
TSnapinColumn = class;
TResultItem = class;
TResultItems = class;

TSnapinDataResultRename = procedure (sender : TObject; const newName : string; var allowRename : boolean) of object;
TSnapinDataResultDelete = procedure (sender : TObject; var allowDelete : boolean) of object;
TSnapinDataResultProperties = procedure (sender : TObject; var Changed : boolean) of object;
TSnapinDataScopeShow = procedure (sender : TObject; selecting : boolean) of object;

TSnapinData = class (TComponent)
private
  FScopeItem : TScopeItem;
  FScopeSmallImages: TImageList;
  FParent: TObject;
  FOnHelp: TNotifyEvent;
  FScopeLargeImages: TImageList;
  FResultSmallImages: TImageList;
  FResultLargeImages: TImageList;
  FOnResultDblClick: TNotifyEvent;
  FOnResultDelete: TSnapinDataResultDelete;
  FOnResultProperties: TSnapinDataResultProperties;
  FOnResultRename: TSnapinDataResultRename;
  FVersionInfo : TVersionInfo;
  fOnScopeShow: TSnapinDataScopeShow;
  fOnScopeExpand: TNotifyEvent;

  procedure SetScopeSmallImages(const Value: TImageList);
  procedure SetParent(const Value: TObject);
  procedure SetScopeLargeImages(const Value: TImageList);
  procedure SetResultLargeImages(const Value: TImageList);
  procedure SetResultSmallImages(const Value: TImageList);
  function GetHandle: HWND;
  function GetProvider: string;
  function GetDescription : string;
  function GetFileVersion: string;
  function GetProductName: string;
protected
public
  constructor Create (AOwner : TComponent); override;
  destructor Destroy; override;
  property Parent : TObject read FParent write SetParent;
  procedure HelpCommand;
  property Handle : HWND read GetHandle;

  property Provider : string read GetProvider;
  property FileDescription : string read GetDescription;
  property FileVersion : string read GetFileVersion;
  property ProductName : string read GetProductName;

published
  property ScopeItem : TScopeItem read FScopeItem write FScopeItem;
  property ScopeSmallImages : TImageList read FScopeSmallImages write SetScopeSmallImages;
  property ScopeLargeImages : TImageList read FScopeLargeImages write SetScopeLargeImages;
  property ResultSmallImages : TImageList read FResultSmallImages write SetResultSmallImages;
  property ResultLargeImages : TImageList read FResultLargeImages write SetResultLargeImages;
  property OnHelp : TNotifyEvent read FOnHelp write FOnHelp;
  property OnResultDblClick : TNotifyEvent read FOnResultDblClick write FOnResultDblClick;
  property OnResultDelete : TSnapinDataResultDelete read FOnResultDelete write FOnResultDelete;
  property OnResultProperties : TSnapinDataResultProperties read FOnResultProperties write FOnResultProperties;
  property OnResultRename : TSnapinDataResultRename read FOnResultRename write FOnResultRename;
  property OnScopeShow : TSnapinDataScopeShow read fOnScopeShow write FOnScopeShow;
  property OnScopeExpand : TNotifyEvent read fOnScopeExpand write fOnScopeExpand;
end;

TScopeItem = class (TCollectionItem)
private
  fText: string;
  FData: pointer;
  FSnapinData : TSnapinData;
  FHasChildren : boolean;
  FViewTypeHTML: string;
  FViewTypeGUID: string;
  FGUIDValid : boolean;
  FImageIndex: Integer;
  FOpenImageIndex : Integer;
  FColumns: TSnapinColumns;
  FContextMenu: TPopupMenu;
  FResultItemsContextMenu : TPopupMenu;
  FScopeItems: TScopeItems;
  FResultItems: TResultItems;
  procedure SetText(const Value: string);
  procedure SetData(const Value: pointer);
  function GetHasChildren: Boolean;
  procedure SetHasChildren(const Value: Boolean);
  function GetViewType: TViewType;
  procedure SetViewTypeGUID(const Value: string);
  procedure SetViewTypeHTML(const Value: string);
  procedure SetImageIndex(const Value: Integer);
  procedure SetColumns(const Value: TSnapinColumns);
  procedure SetContextMenu(const Value: TPopupMenu);
  procedure SetResultItemsContextMenu(const Value: TPopupMenu);
    function GetText: string;
    procedure SetOpenImageIndex(const Value: Integer);
protected
  procedure AssignTo (dest : TPersistent); override;
  function GetOwner : TPersistent; override;
public
  constructor Create (ACollection : TCollection); override;
  destructor Destroy; override;
  procedure DeleteChildren;
  property Data : pointer read FData write SetData;
  property HasChildren: Boolean read GetHasChildren write SetHasChildren;
  procedure HelpCommand;
  property SnapinData : TSnapinData read fSnapinData;
  procedure Refresh;
published
  property Text : string read GetText write SetText;
  property ViewType : TViewType read GetViewType;
  property ViewTypeHTML : string read FViewTypeHTML write SetViewTypeHTML;
  property ViewTypeGUID : string read FViewTypeGUID write SetViewTypeGUID;
  property ImageIndex : Integer read FImageIndex write SetImageIndex default -1;
  property OpenImageIndex : Integer read FOpenImageIndex write SetOpenImageIndex default -1;
  property Columns : TSnapinColumns read FColumns write SetColumns;
  property ContextMenu : TPopupMenu read FContextMenu write SetContextMenu;
  property ResultItemsContextMenu : TPopupMenu read FResultItemsContextMenu write SetResultItemsContextMenu;
  property ScopeItems : TScopeItems read FScopeItems write FScopeItems;
  property ResultItems : TResultItems read FResultItems write FResultItems;
end;

TScopeItems = class (TOwnedCollection)
private
  fSnapinData : TSnapinData;
  function GetItem(i: Integer): TScopeItem;
public
  property Items [i : Integer] : TScopeItem read GetItem; default;
  function Add : TScopeItem;
published
end;

TSnapinColumns = class (TOwnedCollection)
private
  function GetItem(i: Integer): TSnapinColumn;
public
  property Items [i : Integer] : TSnapinColumn read GetItem; default;
  function Add : TSnapinColumn;
end;

TSnapinColumn = class (TCollectionItem)
private
  FWidth: Integer;
  FCaption: string;
  FAlignment: TAlignment;
  procedure SetAlignment(const Value: TAlignment);
  procedure SetCaption(const Value: string);
  procedure SetWidth(const Value: Integer);
published
  property Caption : string read FCaption write SetCaption;
  property Width : Integer read FWidth write SetWidth;
  property Alignment : TAlignment read FAlignment write SetAlignment;
end;

TResultItem = class (TCollectionItem)
private
  FText: string;
  FSubItems: TStrings;
  FImageIndex: Integer;
  fItemID: Integer;
  procedure SetText(const Value: string);
  function GetScopeItem: TScopeItem;

public
  constructor Create (ACollection : TCollection); override;
  destructor Destroy; override;
  property SubItems : TStrings read FSubItems;
  property ScopeItem : TScopeItem read GetScopeItem;
  property itemID : Integer read fItemID write fItemID;
  procedure Refresh;
published
  property Text : string read FText write SetText;
  property ImageIndex : Integer read FImageIndex write FImageIndex default -1;
end;

TResultItems = class (TOwnedCollection)
private
  fUpdateCount : Integer;
  function GetItem(i: Integer): TResultItem;
  function GetScopeItem: TScopeItem;
public
  property Items [i : Integer] : TResultItem read GetItem; default;
  function Add : TResultItem;
  property ScopeItem : TScopeItem read GetScopeItem;
  procedure BeginUpdate; override;
  procedure EndUpdate; override;
end;

procedure Register;

implementation

uses Snapins;

{ TSnapinData }

constructor TSnapinData.Create (aOwner : TComponent);
begin
  inherited Create (aowner);
  FScopeItem := TScopeItem.Create (TCollection (self));
  FVersionInfo := TVersionInfo.Create (HInstance);
end;

destructor TSnapinData.Destroy;
begin
  FScopeItem.Free;
  FVersionInfo.Free;
  inherited Destroy
end;

procedure TSnapinData.HelpCommand;
begin
  if Assigned (FOnHelp) then
    OnHelp (self);
end;

procedure TSnapinData.SetParent(const Value: TObject);
begin
  FParent := Value as TSnapinComponentData;
end;

procedure TSnapinData.SetScopeSmallImages(const Value: TImageList);
var
  parent : TSnapinComponentData;
begin
  FScopeSmallImages := Value;
  parent := TSnapinComponentData (fParent);
  if Assigned (parent) then
    parent.SetScopeImages
end;

procedure TSnapinData.SetScopeLargeImages(const Value: TImageList);
var
  parent : TSnapinComponentData;
begin
  FScopeLargeImages := Value;
  parent := TSnapinComponentData (fParent);
  if Assigned (parent) then
    parent.SetScopeImages
end;

procedure TSnapinData.SetResultLargeImages(const Value: TImageList);
begin
  FResultLargeImages := Value;
end;

procedure TSnapinData.SetResultSmallImages(const Value: TImageList);
begin
  FResultSmallImages := Value;
end;

function TSnapinData.GetHandle: HWND;
var
  parent : TSnapinComponentData;
begin
  parent := TSnapinComponentData (fParent);
  if Assigned (parent) then
    result := parent.WindowHandle
  else
    result := 0;
end;

function TSnapinData.GetProvider: string;
begin
  result := FVersionInfo.KeyValue ['CompanyName'];
end;

function TSnapinData.GetDescription: string;
begin
  result := FVersionInfo.KeyValue ['FileDescription'];
end;

function TSnapinData.GetFileVersion: string;
begin
  result := FVersionInfo.KeyValue ['FileVersion'];
end;

function TSnapinData.GetProductName: string;
begin
  result := FVersionInfo.KeyValue ['ProductName'];
end;

{ TScopeItem }
constructor TScopeItem.Create (ACollection : TCollection);
begin
  if TObject (ACollection) is TSnapinData then
  begin
    FSnapinData := TSnapinData (ACollection);
    ACollection := Nil
  end;
  inherited Create (ACollection);

  if Assigned (ACollection) then
    FSnapinData := TScopeItems (ACollection).FSnapinData;

  FImageIndex := -1;
  FOpenImageIndex := -1;
  FColumns := TSnapinColumns.Create (self, TSnapinColumn);
  FScopeItems := TScopeItems.Create (self, TScopeItem);
  FResultItems := TResultItems.Create (self, TResultItem);
  FScopeItems.fSnapinData := FSnapinData
end;

procedure TScopeItem.DeleteChildren;
begin
  fHasChildren := False;
  FScopeItems.Clear
end;

destructor TScopeItem.Destroy;
begin
  inherited Destroy
end;

function TScopeItem.GetHasChildren: Boolean;
begin
  result := FHasChildren or (FScopeItems.Count > 0)
end;

function TScopeItem.GetViewType: TViewType;
begin
  if FGUIDValid then
    result := vtGUID
  else
    if FViewTypeHTML <> '' then
      result := vtHTML
    else
      result := vtListView
end;

procedure TScopeItem.SetHasChildren(const Value: Boolean);
begin
  if value <> HasChildren then
    FHasChildren := True;
end;

procedure TScopeItem.SetColumns(const Value: TSnapinColumns);
begin
  FColumns := Value;
end;

procedure TScopeItem.SetContextMenu(const Value: TPopupMenu);
begin
  FContextMenu := Value;
end;

procedure TScopeItem.SetResultItemsContextMenu(const Value: TPopupMenu);
begin
  FResultItemsContextMenu := Value;
end;

procedure TScopeItem.SetData(const Value: pointer);
begin
  FData := Value;
end;

procedure TScopeItem.SetImageIndex(const Value: Integer);
begin
  FImageIndex := Value;
end;

procedure TScopeItem.SetText(const Value: string);
begin
  fText := Value;
end;

procedure TScopeItem.SetViewTypeGUID(const Value: string);
begin
  FGUIDValid := True;
  FViewTypeGUID := Value;
  FViewTypeHTML := '';
end;

procedure TScopeItem.SetViewTypeHTML(const Value: string);
begin
  FViewTypeHTML := Value;
  FGUIDValid := False
end;

procedure TScopeItem.AssignTo(dest: TPersistent);
begin
end;

function TScopeItem.GetOwner: TPersistent;
begin
  result := FSnapinData;
end;


procedure TScopeItem.HelpCommand;
begin
 if Assigned (fSnapinData.FOnHelp) then
   fSnapinData.OnHelp (self);
end;

procedure TScopeItem.Refresh;
var
  parent : TSnapinComponentData;
begin
  parent := TSnapinComponentData (SnapinData.Parent);
  if Assigned (parent) then
    parent.Update (self);
end;

function TScopeItem.GetText: string;
begin
  result := fText
end;

procedure TScopeItem.SetOpenImageIndex(const Value: Integer);
begin
  FOpenImageIndex := Value;
end;

{ TSnapinColumn }

procedure TSnapinColumn.SetAlignment(const Value: TAlignment);
begin
  FAlignment := Value;
end;

procedure TSnapinColumn.SetCaption(const Value: string);
begin
  FCaption := Value;
end;

procedure TSnapinColumn.SetWidth(const Value: Integer);
begin
  FWidth := Value;
end;

{ TSnapinColumns }

function TSnapinColumns.Add: TSnapinColumn;
begin
  result := inherited Add as TSnapinColumn

end;

function TSnapinColumns.GetItem(i: Integer): TSnapinColumn;
begin
  result := inherited Items [i] as TSnapinColumn;
end;

procedure Register;
begin
  RegisterComponents ('Snapin', [TSnapinData])
end;

{ TScopeItems }

function TScopeItems.Add: TScopeItem;
begin
  result := inherited Add as TScopeItem;
end;

function TScopeItems.GetItem(i: Integer): TScopeItem;
begin
  result := inherited Items [i] as TScopeItem;
end;

{ TResultItem }

constructor TResultItem.Create(ACollection: TCollection);
begin
  inherited Create (ACollection);
  FSubItems := TStringList.Create;
  FImageIndex := -1
end;

destructor TResultItem.Destroy;
begin
  FSubItems.Free;
  inherited Destroy;
end;

function TResultItem.GetScopeItem: TScopeItem;
begin
  result := (Collection as TResultItems).ScopeItem;
end;

procedure TResultItem.Refresh;
var
  parent : TSnapinComponentData;
begin
  parent := TSnapinComponentData (ScopeItem.SnapinData.fParent);
  if Assigned (parent) then
    parent.Update (self)
end;

procedure TResultItem.SetText(const Value: string);
begin
  FText := Value;
  Refresh
end;

{ TResultItems }

function TResultItems.Add: TResultItem;
begin
  result := inherited Add as TResultItem;
end;

procedure TResultItems.BeginUpdate;
begin
  Inc (fUpdateCount);
  inherited BeginUpdate
end;

procedure TResultItems.EndUpdate;
begin
  Dec (fUpdateCount);
  if fUpdateCount = 0 then
    ScopeItem.Refresh;
  inherited EndUpdate;
end;

function TResultItems.GetItem(i: Integer): TResultItem;
begin
  result := inherited Items [i] as TResultItem;
end;

function TResultItems.GetScopeItem: TScopeItem;
begin
  result := GetOwner as TScopeItem
end;

end.
