unit N3ServerMonitorMainImpl;

{$WARN SYMBOL_PLATFORM OFF}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ActiveX, AxCtrls, N3ServerMonitor_TLB, StdVcl, StdCtrls, ExtCtrls, MJTrayIcon,
  ComCtrls;

type
  TN3ServerMonitorMain = class(TActiveForm, IN3ServerMonitorMain)
    MJTrayIcon: TMJTrayIcon;
    Timer: TTimer;
    Panel: TPanel;
    ProcessList: TListView;
    Button_CloseProcess: TButton;
    Button_KillProcess: TButton;
    procedure TimerTimer(Sender: TObject);
    procedure Button_CloseProcessClick(Sender: TObject);
    procedure Button_KillProcessClick(Sender: TObject);
    procedure ActiveFormCreate(Sender: TObject);
    procedure FormResize(Sender: TObject);
  private
    { Private declarations }
    FEvents: IN3ServerMonitorMainEvents;
    procedure ActivateEvent(Sender: TObject);
    procedure ClickEvent(Sender: TObject);
    procedure CreateEvent(Sender: TObject);
    procedure DblClickEvent(Sender: TObject);
    procedure DeactivateEvent(Sender: TObject);
    procedure DestroyEvent(Sender: TObject);
    procedure KeyPressEvent(Sender: TObject; var Key: Char);
    procedure MouseEnterEvent(Sender: TObject);
    procedure MouseLeaveEvent(Sender: TObject);
    procedure PaintEvent(Sender: TObject);
  protected
    { Protected declarations }
    procedure DefinePropertyPages(DefinePropertyPage: TDefinePropertyPage); override;
    procedure EventSinkChanged(const EventSink: IUnknown); override;
    function Get_Active: WordBool; safecall;
    function Get_AlignDisabled: WordBool; safecall;
    function Get_AlignWithMargins: WordBool; safecall;
    function Get_AutoScroll: WordBool; safecall;
    function Get_AutoSize: WordBool; safecall;
    function Get_AxBorderStyle: TxActiveFormBorderStyle; safecall;
    function Get_Caption: WideString; safecall;
    function Get_Color: OLE_COLOR; safecall;
    function Get_DockSite: WordBool; safecall;
    function Get_DoubleBuffered: WordBool; safecall;
    function Get_DropTarget: WordBool; safecall;
    function Get_Enabled: WordBool; safecall;
    function Get_ExplicitHeight: Integer; safecall;
    function Get_ExplicitLeft: Integer; safecall;
    function Get_ExplicitTop: Integer; safecall;
    function Get_ExplicitWidth: Integer; safecall;
    function Get_Font: IFontDisp; safecall;
    function Get_HelpFile: WideString; safecall;
    function Get_KeyPreview: WordBool; safecall;
    function Get_MouseInClient: WordBool; safecall;
    function Get_PixelsPerInch: Integer; safecall;
    function Get_PopupMode: TxPopupMode; safecall;
    function Get_PrintScale: TxPrintScale; safecall;
    function Get_Scaled: WordBool; safecall;
    function Get_ScreenSnap: WordBool; safecall;
    function Get_SnapBuffer: Integer; safecall;
    function Get_UseDockManager: WordBool; safecall;
    function Get_Visible: WordBool; safecall;
    function Get_VisibleDockClientCount: Integer; safecall;
    procedure _Set_Font(var Value: IFontDisp); safecall;
    procedure Set_AlignWithMargins(Value: WordBool); safecall;
    procedure Set_AutoScroll(Value: WordBool); safecall;
    procedure Set_AutoSize(Value: WordBool); safecall;
    procedure Set_AxBorderStyle(Value: TxActiveFormBorderStyle); safecall;
    procedure Set_Caption(const Value: WideString); safecall;
    procedure Set_Color(Value: OLE_COLOR); safecall;
    procedure Set_DockSite(Value: WordBool); safecall;
    procedure Set_DoubleBuffered(Value: WordBool); safecall;
    procedure Set_DropTarget(Value: WordBool); safecall;
    procedure Set_Enabled(Value: WordBool); safecall;
    procedure Set_Font(const Value: IFontDisp); safecall;
    procedure Set_HelpFile(const Value: WideString); safecall;
    procedure Set_KeyPreview(Value: WordBool); safecall;
    procedure Set_PixelsPerInch(Value: Integer); safecall;
    procedure Set_PopupMode(Value: TxPopupMode); safecall;
    procedure Set_PrintScale(Value: TxPrintScale); safecall;
    procedure Set_Scaled(Value: WordBool); safecall;
    procedure Set_ScreenSnap(Value: WordBool); safecall;
    procedure Set_SnapBuffer(Value: Integer); safecall;
    procedure Set_UseDockManager(Value: WordBool); safecall;
    procedure Set_Visible(Value: WordBool); safecall;
  public
    { Public declarations }
    procedure Initialize; override;
  end;

type
  TWindowsVersion = (wvUnknown, wvWin95, wvWin95OSR2, wvWin98, wvWin98SE, wvWinME,
                     wvWinNT31, wvWinNT35, wvWinNT351, wvWinNT4, wvWin2000, wvWinXP, wvWin2003);

var
  IsWin95: Boolean = False;
  IsWin95OSR2: Boolean = False;
  IsWin98: Boolean = False;
  IsWin98SE: Boolean = False;
  IsWinME: Boolean = False;
  IsWinNT: Boolean = False;
  IsWinNT3: Boolean = False;
  IsWinNT31: Boolean = False;
  IsWinNT35: Boolean = False;
  IsWinNT351: Boolean = False;
  IsWinNT4: Boolean = False;
  IsWin2K: Boolean = False;
  IsWinXP: Boolean = False;
  IsWin2003: Boolean = False;

  ProcessorCount: Cardinal = 0;
  AllocGranularity: Cardinal = 0;
  PageSize: Cardinal = 0;

resourcestring
  RsSystemProcess = 'System Process';
  RsSystemIdleProcess = 'System Idle Process';

implementation

uses
  ComObj, ComServ, ShlObj, TLHelp32, PsApi, unitVersionInfo, Unit1;

{$R *.DFM}

var
  KernelVersionHi: DWORD;

function GetWindowsVersion: TWindowsVersion;
var
  TrimmedWin32CSDVersion: string;
begin
  Result := wvUnknown;
  TrimmedWin32CSDVersion := Trim(Win32CSDVersion);
  case Win32Platform of
    VER_PLATFORM_WIN32_WINDOWS:
      case Win32MinorVersion of
        0..9:
          if (TrimmedWin32CSDVersion = 'B') or (TrimmedWin32CSDVersion = 'C') then
            Result := wvWin95OSR2
          else
            Result := wvWin95;
        10..89:
          // On Windows ME Win32MinorVersion can be 10 (indicating Windows 98
          // under certain circumstances (image name is setup.exe). Checking
          // the kernel version is one way of working around that.
          if KernelVersionHi = $0004005A then // 4.90.x.x
            Result := wvWinME
          else
          if TrimmedWin32CSDVersion = 'A' then
            Result := wvWin98SE
          else
            Result := wvWin98;
        90:
          Result := wvWinME;
      end;
    VER_PLATFORM_WIN32_NT:
      case Win32MajorVersion of
        3:
          case Win32MinorVersion of
            1:
              Result := wvWinNT31;
            5:
              Result := wvWinNT35;
            51:
              Result := wvWinNT351;
          end;
        4:
          Result := wvWinNT4;
        5:
          case Win32MinorVersion of
            0:
              Result := wvWin2000;
            1:
              Result := wvWinXP;
            else
              Result := wvWin2003;
          end;
      end;
  end;
end;

{ TN3ServerMonitorMain }

procedure TN3ServerMonitorMain.DefinePropertyPages(DefinePropertyPage: TDefinePropertyPage);
begin
  { Define property pages here.  Property pages are defined by calling
    DefinePropertyPage with the class id of the page.  For example,
      DefinePropertyPage(Class_N3ServerMonitorMainPage); }
end;

procedure TN3ServerMonitorMain.EventSinkChanged(const EventSink: IUnknown);
begin
  FEvents := EventSink as IN3ServerMonitorMainEvents;
  inherited EventSinkChanged(EventSink);
end;

procedure TN3ServerMonitorMain.FormResize(Sender: TObject);
begin
  if (Width = 0) or (Height = 0) then
    Panel.Align := alNone
  else
    Panel.Align := alClient;
end;

procedure TN3ServerMonitorMain.Initialize;
begin
  inherited Initialize;
  OnActivate := ActivateEvent;
  OnClick := ClickEvent;
  OnCreate := CreateEvent;
  OnDblClick := DblClickEvent;
  OnDeactivate := DeactivateEvent;
  OnDestroy := DestroyEvent;
  OnKeyPress := KeyPressEvent;
  OnMouseEnter := MouseEnterEvent;
  OnMouseLeave := MouseLeaveEvent;
  OnPaint := PaintEvent;
end;

function TN3ServerMonitorMain.Get_Active: WordBool;
begin
  Result := Active;
end;

function TN3ServerMonitorMain.Get_AlignDisabled: WordBool;
begin
  Result := AlignDisabled;
end;

function TN3ServerMonitorMain.Get_AlignWithMargins: WordBool;
begin
  Result := AlignWithMargins;
end;

function TN3ServerMonitorMain.Get_AutoScroll: WordBool;
begin
  Result := AutoScroll;
end;

function TN3ServerMonitorMain.Get_AutoSize: WordBool;
begin
  Result := AutoSize;
end;

function TN3ServerMonitorMain.Get_AxBorderStyle: TxActiveFormBorderStyle;
begin
  Result := Ord(AxBorderStyle);
end;

function TN3ServerMonitorMain.Get_Caption: WideString;
begin
  Result := WideString(Caption);
end;

function TN3ServerMonitorMain.Get_Color: OLE_COLOR;
begin
  Result := OLE_COLOR(Color);
end;

function TN3ServerMonitorMain.Get_DockSite: WordBool;
begin
  Result := DockSite;
end;

function TN3ServerMonitorMain.Get_DoubleBuffered: WordBool;
begin
  Result := DoubleBuffered;
end;

function TN3ServerMonitorMain.Get_DropTarget: WordBool;
begin
  Result := DropTarget;
end;

function TN3ServerMonitorMain.Get_Enabled: WordBool;
begin
  Result := Enabled;
end;

function TN3ServerMonitorMain.Get_ExplicitHeight: Integer;
begin
  Result := ExplicitHeight;
end;

function TN3ServerMonitorMain.Get_ExplicitLeft: Integer;
begin
  Result := ExplicitLeft;
end;

function TN3ServerMonitorMain.Get_ExplicitTop: Integer;
begin
  Result := ExplicitTop;
end;

function TN3ServerMonitorMain.Get_ExplicitWidth: Integer;
begin
  Result := ExplicitWidth;
end;

function TN3ServerMonitorMain.Get_Font: IFontDisp;
begin
  GetOleFont(Font, Result);
end;

function TN3ServerMonitorMain.Get_HelpFile: WideString;
begin
  Result := WideString(HelpFile);
end;

function TN3ServerMonitorMain.Get_KeyPreview: WordBool;
begin
  Result := KeyPreview;
end;

function TN3ServerMonitorMain.Get_MouseInClient: WordBool;
begin
  Result := MouseInClient;
end;

function TN3ServerMonitorMain.Get_PixelsPerInch: Integer;
begin
  Result := PixelsPerInch;
end;

function TN3ServerMonitorMain.Get_PopupMode: TxPopupMode;
begin
  Result := Ord(PopupMode);
end;

function TN3ServerMonitorMain.Get_PrintScale: TxPrintScale;
begin
  Result := Ord(PrintScale);
end;

function TN3ServerMonitorMain.Get_Scaled: WordBool;
begin
  Result := Scaled;
end;

function TN3ServerMonitorMain.Get_ScreenSnap: WordBool;
begin
  Result := ScreenSnap;
end;

function TN3ServerMonitorMain.Get_SnapBuffer: Integer;
begin
  Result := SnapBuffer;
end;

function TN3ServerMonitorMain.Get_UseDockManager: WordBool;
begin
  Result := UseDockManager;
end;

function TN3ServerMonitorMain.Get_Visible: WordBool;
begin
  Result := Visible;
end;

function TN3ServerMonitorMain.Get_VisibleDockClientCount: Integer;
begin
  Result := VisibleDockClientCount;
end;

procedure TN3ServerMonitorMain._Set_Font(var Value: IFontDisp);
begin
  SetOleFont(Font, Value);
end;

procedure TN3ServerMonitorMain.ActivateEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnActivate;
end;

procedure TN3ServerMonitorMain.ActiveFormCreate(Sender: TObject);
begin
  OnResize := FormResize;
{  Windows.Beep(2500, 100);
  Show;}
{  ShowMessage(IntToStr(Width) + 'x' + IntToStr(Height));
  Panel.Width := Width;
  Panel.Height := Height;}

//  Panel.Align := alClient;
end;

procedure TN3ServerMonitorMain.Button_CloseProcessClick(Sender: TObject);

  function EnumWindowsProc(Wnd: HWND; ProcessID: DWORD): Boolean; stdcall;
  var
    PID: DWORD;
  begin
    GetWindowThreadProcessId(Wnd, @PID);
    if ProcessID = PID then
      PostMessage(Wnd, WM_CLOSE, 0, 0);
    Result := True;
  end;

var
  Item: TListItem;
  ProcessID: Cardinal;
  HProc: THandle;
begin
  ClickEvent



  Exit;




  Item := ProcessList.Selected;
  if not Assigned(Item) then
    Exit;
  ProcessID := StrToInt(Item.Caption);
  HProc := OpenProcess(SYNCHRONIZE or PROCESS_TERMINATE, False, ProcessID);
  if HProc <> 0 then
  try
    EnumWindows(@EnumWindowsProc, LPARAM(ProcessID));
    if WaitForSingleObject(HProc, 5000) <> WAIT_OBJECT_0 then
      MessageBox(0, '  !', nil, MB_ICONERROR);
  finally
    CloseHandle(HProc);
  end;
end;

procedure TN3ServerMonitorMain.Button_KillProcessClick(Sender: TObject);
var
  Item: TListItem;
  ProcessID: Cardinal;
  HProc: THandle;
begin
  Item := ProcessList.Selected;
  if not Assigned(Item) then
    Exit;
  ProcessID := StrToInt(Item.Caption);
  HProc := OpenProcess(PROCESS_TERMINATE, False, ProcessID);
  if HProc <> 0 then
  try
    if not TerminateProcess(HProc, 0) then
      MessageBox(0, '  !', nil, MB_ICONERROR);
  finally
    CloseHandle(HProc);
  end;
end;

procedure TN3ServerMonitorMain.ClickEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnClick;
end;

procedure TN3ServerMonitorMain.CreateEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnCreate;
end;

procedure TN3ServerMonitorMain.DblClickEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnDblClick;
end;

procedure TN3ServerMonitorMain.DeactivateEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnDeactivate;
end;

procedure TN3ServerMonitorMain.DestroyEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnDestroy;
end;

procedure TN3ServerMonitorMain.KeyPressEvent(Sender: TObject; var Key: Char);
var
  TempKey: Smallint;
begin
  TempKey := Smallint(Key);
  if FEvents <> nil then FEvents.OnKeyPress(TempKey);
  Key := Char(TempKey);
end;

procedure TN3ServerMonitorMain.MouseEnterEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnMouseEnter;
end;

procedure TN3ServerMonitorMain.MouseLeaveEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnMouseLeave;
end;

procedure TN3ServerMonitorMain.PaintEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnPaint;
end;

procedure TN3ServerMonitorMain.Set_AlignWithMargins(Value: WordBool);
begin
  AlignWithMargins := Value;
end;

procedure TN3ServerMonitorMain.Set_AutoScroll(Value: WordBool);
begin
  AutoScroll := Value;
end;

procedure TN3ServerMonitorMain.Set_AutoSize(Value: WordBool);
begin
  AutoSize := Value;
end;

procedure TN3ServerMonitorMain.Set_AxBorderStyle(
  Value: TxActiveFormBorderStyle);
begin
  AxBorderStyle := TActiveFormBorderStyle(Value);
end;

procedure TN3ServerMonitorMain.Set_Caption(const Value: WideString);
begin
  Caption := TCaption(Value);
end;

procedure TN3ServerMonitorMain.Set_Color(Value: OLE_COLOR);
begin
  Color := TColor(Value);
end;

procedure TN3ServerMonitorMain.Set_DockSite(Value: WordBool);
begin
  DockSite := Value;
end;

procedure TN3ServerMonitorMain.Set_DoubleBuffered(Value: WordBool);
begin
  DoubleBuffered := Value;
end;

procedure TN3ServerMonitorMain.Set_DropTarget(Value: WordBool);
begin
  DropTarget := Value;
end;

procedure TN3ServerMonitorMain.Set_Enabled(Value: WordBool);
begin
  Enabled := Value;
end;

procedure TN3ServerMonitorMain.Set_Font(const Value: IFontDisp);
begin
  SetOleFont(Font, Value);
end;

procedure TN3ServerMonitorMain.Set_HelpFile(const Value: WideString);
begin
  HelpFile := string(Value);
end;

procedure TN3ServerMonitorMain.Set_KeyPreview(Value: WordBool);
begin
  KeyPreview := Value;
end;

procedure TN3ServerMonitorMain.Set_PixelsPerInch(Value: Integer);
begin
  PixelsPerInch := Value;
end;

procedure TN3ServerMonitorMain.Set_PopupMode(Value: TxPopupMode);
begin
  PopupMode := TPopupMode(Value);
end;

procedure TN3ServerMonitorMain.Set_PrintScale(Value: TxPrintScale);
begin
  PrintScale := TPrintScale(Value);
end;

procedure TN3ServerMonitorMain.Set_Scaled(Value: WordBool);
begin
  Scaled := Value;
end;

procedure TN3ServerMonitorMain.Set_ScreenSnap(Value: WordBool);
begin
  ScreenSnap := Value;
end;

procedure TN3ServerMonitorMain.Set_SnapBuffer(Value: Integer);
begin
  SnapBuffer := Value;
end;

procedure TN3ServerMonitorMain.Set_UseDockManager(Value: WordBool);
begin
  UseDockManager := Value;
end;

procedure TN3ServerMonitorMain.Set_Visible(Value: WordBool);
begin
  Visible := Value;
end;

type
  TProcType = (ptNoCheck, ptUnknown, ptServer, ptClient, ptAdmin, ptUnknownEntry);

  TProcListEntry = record
    PID: Cardinal;
    Name: String;
    ProcType: TProcType;
  end;
  TProcList = array of TProcListEntry;

var
  CashProcList: TProcList;

function RunningProcessesList(List: TListView): Boolean;

var
  ProcList: TProcList;
  li: Integer;
  Item: TListItem;

  function ProcessFileName(PID: DWORD): string;
  var
    Handle: THandle;
  begin
    Result := '';
    Handle := OpenProcess(PROCESS_QUERY_INFORMATION or PROCESS_VM_READ, False, PID);
    if Handle <> 0 then
    try
      SetLength(Result, MAX_PATH);
      if GetModuleFileNameEx(Handle, 0, PChar(Result), MAX_PATH) > 0 then
        SetLength(Result, StrLen(PChar(Result)))
      else
        Result := '';
    finally
      CloseHandle(Handle);
    end;
  end;

  function BuildListTH: Boolean;
  var
    SnapProcHandle: THandle;
    ProcEntry: TProcessEntry32;
    NextProc: Boolean;
    FileName: string;
  begin
    SnapProcHandle := CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
    Result := (SnapProcHandle <> INVALID_HANDLE_VALUE);
    if Result then
    try
      ProcEntry.dwSize := SizeOf(ProcEntry);
      NextProc := Process32First(SnapProcHandle, ProcEntry);
      while NextProc do
      begin
        if ProcEntry.th32ProcessID = 0 then
        begin
          // PID 0 is always the "System Idle Process" but this name cannot be
          // retrieved from the system and has to be fabricated.
          FileName := RsSystemIdleProcess;
        end
        else
        begin
          FileName := ProcessFileName(ProcEntry.th32ProcessID);
          if FileName = '' then
            FileName := ProcEntry.szExeFile;
        end;
        li := Length(ProcList);
        SetLength(ProcList, li + 1);
        ProcList[li].PID := ProcEntry.th32ProcessID;
        ProcList[li].Name := FileName;
        NextProc := Process32Next(SnapProcHandle, ProcEntry);
      end;
    finally
      CloseHandle(SnapProcHandle);
    end;
  end;

  function BuildListPS: Boolean;
  var
    PIDs: array [0..1024] of DWORD;
    Needed: DWORD;
    I: Integer;
    FileName: string;
  begin
    Result := EnumProcesses(@PIDs, SizeOf(PIDs), Needed);
    if Result then
    begin
      for I := 0 to (Needed div SizeOf(DWORD)) - 1 do
      begin
        case PIDs[I] of
          0:
            // PID 0 is always the "System Idle Process" but this name cannot be
            // retrieved from the system and has to be fabricated.
            FileName := RsSystemIdleProcess;
          2:
            // On NT 4 PID 2 is the "System Process" but this name cannot be
            // retrieved from the system and has to be fabricated.
            if IsWinNT4 then
              FileName := RsSystemProcess
            else
              FileName := ProcessFileName(PIDs[I]);
          8:
            // On Win2K PID 8 is the "System Process" but this name cannot be
            // retrieved from the system and has to be fabricated.
            if IsWin2k or IsWinXP then
              FileName := RsSystemProcess
            else
              FileName := ProcessFileName(PIDs[I]);
        else
          FileName := ProcessFileName(PIDs[I]);
        end;
        if FileName <> '' then
        begin
          li := Length(ProcList);
          SetLength(ProcList, li + 1);
          ProcList[li].PID := PIDs[I];
          ProcList[li].Name := FileName;
        end;
      end;
    end;
  end;

  procedure ProcListSort();

    function SCompare(I, P: Integer): Integer;
    var
      V1, V2: Cardinal;
    begin
      V1 := ProcList[I].PID;
      V2 := ProcList[P].PID;
      if V1 = V2 then
        Result := 0
      else if V1 < V2 then
        Result := -1
      else
        Result := 1;
    end;

    procedure ExchangeItems(L, R: Integer);
    var
      T: TProcListEntry;
    begin
      T := ProcList[L];
      ProcList[L] := ProcList[R];
      ProcList[R] := T;
    end;

    procedure QuickSort(L, R: Integer);
    var
      I, J, P: Integer;
    begin
      repeat
        I := L;
        J := R;
        P := (L + R) shr 1;
        repeat
          while SCompare(I, P) < 0 do Inc(I);
          while SCompare(J, P) > 0 do Dec(J);
          if I <= J then
          begin
            ExchangeItems(I, J);
            if P = I then
              P := J
            else if P = J then
              P := I;
            Inc(I);
            Dec(J);
          end;
        until I > J;
        if L < J then QuickSort(L, J);
        L := I;
      until I >= R;
    end;

  begin
    QuickSort(0, Length(ProcList) - 1);
  end;

  function ProcListSynk: Boolean;

    function GetProcType(FileName: string): TProcType;
    var
      Handle: THandle;
      Size: DWORD;
      V: TVersionInfo;
      Buffer: string;
    begin
      Size := GetFileVersionInfoSize(PChar(FileName), Handle);
      try
        if Size = 0 then
        begin
          Result := ptUnknown;
          Exit;
        end;
        SetLength(Buffer, Size);
        if not GetFileVersionInfo(PChar(FileName), Handle, Size, PChar(Buffer)) then
        begin
          Result := ptUnknown;
          Exit;
        end;
      finally
        CloseHandle(Handle);
      end;
      try
        V := TVersionInfo.Create(PChar(Buffer));
        try
          if V.KeyCount = 0 then
        begin
          Result := ptUnknown;
          Exit;
        end;
{
    'Comments'
    'CompanyName'
    'FileDescription'
    'FileVersion'
    'InternalName'
    'LegalCopyright'
    'LegalTradeMarks'
    'OriginalFilename'
    'ProductName'
    'ProductVersion'
    'SpecialBuild'
    'PrivateBuild'
}
          if Pos('N3Server', FileName) > 0 then
            Result := ptServer
          else if Pos('N3Server', V.KeyValue['OriginalFilename']) > 0 then
            Result := ptServer
          else if Pos('N3Server', V.KeyValue['InternalName']) > 0 then
            Result := ptServer
          else if Pos('Entry', V.KeyValue['CompanyName']) > 0 then
            Result := ptUnknownEntry
          else
            Result := ptUnknown;
        finally
          V.Free();
        end;
      except
        Result := ptUnknown;
      end;
    end;

  var
    I: Integer;
    PID: Cardinal;
    CurrentP: Integer;
    MaxP: Integer;
    IsUpdate: Boolean;
  begin
    IsUpdate := False;
    CurrentP := 0;
    MaxP := Length(CashProcList) - 1;
    for I := 0 to Length(ProcList) - 1 do
    begin
      PID := ProcList[I].PID;
      while (CurrentP <= MaxP) and (PID > CashProcList[CurrentP].PID) do
      begin
        if CashProcList[CurrentP].ProcType <> ptUnknown then
          IsUpdate := True;
        Inc(CurrentP);
      end;
      if (CurrentP <= MaxP) and (PID = CashProcList[CurrentP].PID) then
      begin
        ProcList[I].ProcType := CashProcList[CurrentP].ProcType;
        Inc(CurrentP);
      end
      else begin
        ProcList[I].ProcType := GetProcType(ProcList[I].Name);
        if ProcList[I].ProcType <> ptUnknown then
          IsUpdate := True;
      end;
    end;
    CashProcList := ProcList;
    Result := IsUpdate;
  end;

  procedure ProcListUpdate;
  var
    I: Integer;
  begin
    List.Items.BeginUpdate();
    List.Items.Clear();
    for I := 0 to Length(ProcList) - 1 do
    begin
      if ProcList[I].ProcType = ptUnknown then
        Continue;
      Item := List.Items.Add();
      Item.Caption := IntToStr(ProcList[I].PID);
      Item.SubItems.Add(ExtractFileName(ProcList[I].Name));
      Item.SubItems.Add(ExtractFilePath(ProcList[I].Name));
    end;
    List.Items.EndUpdate();
  end;

begin
  SetLength(ProcList, 0);
  if GetWindowsVersion in [wvWinNT31, wvWinNT35, wvWinNT351, wvWinNT4] then
    Result := BuildListPS
  else
    Result := BuildListTH;
  ProcListSort();
  if ProcListSynk() then
    ProcListUpdate();
end;

procedure TN3ServerMonitorMain.TimerTimer(Sender: TObject);
begin
  RunningProcessesList(ProcessList);
end;

function VersionFixedFileInfo(const FileName: string; var FixedInfo: TVSFixedFileInfo): Boolean;
var
  Size, FixInfoLen: DWORD;
  Handle: THandle;
  Buffer: string;
  FixInfoBuf: PVSFixedFileInfo;
begin
  Result := False;
  Size := GetFileVersionInfoSize(PChar(FileName), Handle);
  if Size > 0 then
  begin
    SetLength(Buffer, Size);
    if GetFileVersionInfo(PChar(FileName), Handle, Size, Pointer(Buffer)) and
      VerQueryValue(Pointer(Buffer), '\', Pointer(FixInfoBuf), FixInfoLen) and
      (FixInfoLen = SizeOf(TVSFixedFileInfo)) then
    begin
      Result := True;
      FixedInfo := FixInfoBuf^;
    end;
  end;
end;

procedure InitSysInfo;
var
  SystemInfo: TSystemInfo;
  Kernel32FileName: string;
  VerFixedFileInfo: TVSFixedFileInfo;
begin
  FillChar(SystemInfo, SizeOf(SystemInfo), #0);
  GetSystemInfo(SystemInfo);
  ProcessorCount := SystemInfo.dwNumberOfProcessors;
  AllocGranularity := SystemInfo.dwAllocationGranularity;
  PageSize := SystemInfo.dwPageSize;

  { Windows version information }

  IsWinNT := Win32Platform = VER_PLATFORM_WIN32_NT;

  Kernel32FileName := GetModuleName(GetModuleHandle(kernel32));
  if (not IsWinNT) and VersionFixedFileInfo(Kernel32FileName, VerFixedFileInfo) then
    KernelVersionHi := VerFixedFileInfo.dwProductVersionMS
  else
    KernelVersionHi := 0;

  case GetWindowsVersion of
    wvUnknown: ;
    wvWin95:
      IsWin95 := True;
    wvWin95OSR2:
      IsWin95OSR2 := True;
    wvWin98:
      IsWin98 := True;
    wvWin98SE:
      IsWin98SE := True;
    wvWinME:
      IsWinME := True;
    wvWinNT31:
      begin
        IsWinNT3 := True;
        IsWinNT31 := True;
      end;
    wvWinNT35:
      begin
        IsWinNT3 := True;
        IsWinNT35 := True;
      end;
    wvWinNT351:
      begin
        IsWinNT3 := True;
        IsWinNT351 := True;
      end;
    wvWinNT4:
      IsWinNT4 := True;
    wvWin2000:
      IsWin2K := True;
    wvWinXP:
      IsWinXP := True;
    wvWin2003:
      IsWin2003 := True;
  end;
end;

initialization
  TActiveFormFactory.Create(
    ComServer,
    TActiveFormControl,
    TN3ServerMonitorMain,
    Class_N3ServerMonitorMain,
    1,
    '',
    OLEMISC_SIMPLEFRAME or OLEMISC_ACTSLIKELABEL,
    tmApartment);
  InitSysInfo();

end.
