(*==========================================================================*
 | unit Snapins                                                             |
 |                                                                          |
 | Wrapper for MMC snapins                                                  |
 |                                                                          |
 | * LIMITATION - Does not support out-of-process snapins.                  |
 |                                                                          |
 | Version  Date     By            Description                              |
 | -------  -------  ------------  -----------------------------------------|
 | 1.0      17/8/99  Colin Wilson  Original                                 |
 *==========================================================================*)
unit Snapins;

interface

uses Windows, Messages, ActiveX, SysUtils, ComObj, Classes, Menus, MMC_TLB,SnapinData, Graphics, CommCtrl, imglist, controls ;

type

HSCOPEITEM = LongInt;

(*--------------------------------------------------------------------------*
 | class TInternal                                                          |
 |                                                                          |
 | Internal objects get sent by MMC in IDataObjects using the               |
 | CCF_SNAPIN_INTERNAL format.  The TObject will either be a TScopeItem or  |
 | a TResultItem object.                                                    |
 *--------------------------------------------------------------------------*)
TInternal = record
  m_type : _DATA_OBJECT_TYPES;        // What context is the data object.
  m_cookie : LongInt;                 // What object the cookie represents
  m_clsid : TGuid;                    // Class ID of who created this data object
  m_object : TObject;                 // Hence won't work with DCOM...
end;
PInternal = ^TInternal;

(*--------------------------------------------------------------------------*
 | class TSnapinComponentData                                               |
 |                                                                          |
 | Snapin 'doc' object                                                      |
 *--------------------------------------------------------------------------*)
TSnapinComponentData = class (TComObject, IComponentData, IExtendPropertySheet, IExtendContextMenu, IPersistStream)
private
  fConsole : IConsole;
  fConsole2 : IConsole2;
  fConsoleNameSpace : IConsoleNameSpace;
  fScopeImageList : IImageList;
  fSnapinData : TSnapinData;
  FInitialized: boolean;
  fIsDirty : boolean;

  function GetNodeForDataObject (const dataObj : IDataObject) : TObject;
  function GetNodeForCookie (cookie : Integer) : TObject;
  function GetWindowHandle: HWND;

protected
  function IComponentData.Initialize = ComponentDataInitialize;
  function IComponentData.CreateComponent = ComponentDataCreateComponent;
  function IComponentData.Notify = ComponentDataNotify;
  function IComponentData.Destroy = ComponentDataDestroy;
  function IComponentData.QueryDataObject = ComponentDataQueryDataObject;
  function IComponentData.GetDisplayInfo = ComponentDataGetDisplayInfo;
  function IComponentData.CompareObjects = ComponentDataCompareObjects;

  function IExtendPropertySheet.CreatePropertyPages = ExtendPropertySheetCreatePropertyPages;
  function IExtendPropertySheet.QueryPagesFor = ExtendPropertySheetQueryPagesFor;

  function IExtendContextMenu.Command = ExtendContextMenuCommand;
  function IExtendContextMenu.AddMenuItems = ExtendContextMenuAddMenuItems;

  function IPersistStream.IsDirty = PersistStreamIsDirty;
  function IPersistStream.Load = PersistStreamLoad;
  function IPersistStream.Save = PersistStreamSave;
  function IPersistStream.GetSizeMax = PersistStreamGetSizeMax;
  function IPersistStream.GetClassID = PersistStreamGetClassID;

  function ComponentDataInitialize(const pUnknown: IUnknown): HResult; stdcall;
  function ComponentDataCreateComponent(out ppComponent: IComponent): HResult; stdcall;
  function ComponentDataNotify(const lpDataObject: IDataObject; event: _MMC_NOTIFY_TYPE; arg: Integer;
                  param: Integer): HResult; stdcall;
  function ComponentDataDestroy: HResult; stdcall;
  function ComponentDataQueryDataObject(cookie: Integer; _type: _DATA_OBJECT_TYPES;
                           out ppDataObject: IDataObject): HResult; stdcall;
  function ComponentDataGetDisplayInfo(var pScopeDataItem: _SCOPEDATAITEM): HResult; stdcall;
  function ComponentDataCompareObjects(const lpDataObjectA: IDataObject; const lpDataObjectB: IDataObject): HResult; stdcall;

  function ExtendPropertySheetCreatePropertyPages(const lpProvider: IPropertySheetCallback; handle: Integer;
                                 const lpIDataObject: IDataObject): HResult; stdcall;
  function ExtendPropertySheetQueryPagesFor(const lpDataObject: IDataObject): HResult; stdcall;

  function PersistStreamIsDirty: HResult; stdcall;
  function PersistStreamLoad (const stm: IStream): HResult; stdcall;
  function PersistStreamSave (const stm: IStream; fClearDirty: BOOL): HResult; stdcall;
  function PersistStreamGetSizeMax (out cbSize: Largeint): HResult; stdcall;
  function PersistStreamGetClassID (out classID: TCLSID): HResult; stdcall;

  function ExtendContextMenuCommand (nCommandID : Integer; const dataObject : IDataObject) : HRESULT; stdcall;
  function ExtendContextMenuAddMenuItems (const dataObject : IDataObject;  const piCallback: IContextMenuCallback; var pInsertionAllowed: Integer) : HRESULT; stdcall;

  function GetSnapinData : TSnapinData; virtual; abstract;

  procedure EnumerateScopePane (dataObject : IDataObject; pParent : HSCOPEITEM);
public
  property SnapinData : TSnapinData read fSnapinData;
  procedure Initialize; override;
  property Initialized : boolean read FInitialized;
  procedure SetScopeImages;
  property WindowHandle : HWND read GetWindowHandle;
  procedure Update (Item : TObject);
end;

(*--------------------------------------------------------------------------*
 | class TSnapinAbout                                                       |
 *--------------------------------------------------------------------------*)
TSnapinAbout = class (TComObject, ISnapinAbout)
private
  fSnapinData : TSnapinData;
  procedure LoadSnapinData;
protected
  function GetSnapinDescription(out lpDescription: PWideChar): HResult; stdcall;
  function GetProvider(out lpName : PWideChar): HResult; stdcall;
  function GetSnapinVersion(out lpVersion: PWideChar): HResult; stdcall;
  function GetSnapinImage(out hAppIcon: wireHICON): HResult; stdcall;
  function GetStaticFolderImage(out hSmallImage: wireHBITMAP; out hSmallImageOpen: wireHBITMAP;
                                out hLargeImage: wireHBITMAP; out cMask: UINT): HResult; stdcall;

  function GetSnapinData : TSnapinData; virtual; abstract;
public

end;

(*--------------------------------------------------------------------------*
 | class TSnapinComponent                                                   |
 *--------------------------------------------------------------------------*)
TSnapinComponent = class (TInterfacedObject, IComponent, IExtendContextMenu, IExtendControlBar, IResultDataCompare, IExtendPropertySheet)
private
  fConsole : IConsole;
  fConsole2 : IConsole2;
  fHeader : IHeaderCtrl;
  fResultData : IResultData;
  fParent : TSnapinComponentData;
  fConsoleVerb : IConsoleVerb;
  fResultImages : IImageList;
  fDeletingItem : boolean;
  fCurrentScopeItem : TScopeItem;
  fInShowing : boolean;

  procedure SetResultImages (useScopeImages : boolean);
  procedure InitializeHeaders (node : TScopeItem);
  procedure EnumerateResultPane (node : TScopeItem);
  procedure HandleStandardVerbs (bDeselectAll : boolean; arg : LongInt; scopeItem : TScopeItem; resultItem : TResultItem);
  procedure DecodeDataObject (dataObject : IDataObject; var scopeItem : TScopeItem; var resultItem : TResultItem);

protected
  function IComponent.Initialize = ComponentInitialize;
  function IComponent.Notify = ComponentNotify;
  function IComponent.Destroy = ComponentDestroy;
  function IComponent.QueryDataObject = ComponentQueryDataObject;
  function IComponent.GetResultViewType = ComponentGetResultViewType;
  function IComponent.GetDisplayInfo = ComponentGetDisplayInfo;
  function IComponent.CompareObjects = ComponentCompareObjects;

  function IExtendContextMenu.Command = ExtendContextMenuCommand;
  function IExtendContextMenu.AddMenuItems = ExtendContextMenuAddMenuItems;

  function IExtendControlbar.SetControlbar = ExtendControlbarSetControlbar;
  function IExtendControlbar.ControlbarNotify = ExtendControlbarControlbarNotify;

  function IResultDataCompare.Compare = ResultDataCompareCompare;

  function IExtendPropertySheet.CreatePropertyPages = ExtendPropertySheetCreatePropertyPages;
  function IExtendPropertySheet.QueryPagesFor = ExtendPropertySheetQueryPagesFor;

  function ComponentInitialize(const lpConsole: IConsole): HResult; stdcall;
  function ComponentNotify(const lpDataObject: IDataObject; event: _MMC_NOTIFY_TYPE; arg: Integer;
                  param: Integer): HResult; stdcall;
  function ComponentDestroy(cookie: Integer): HResult; stdcall;
  function ComponentQueryDataObject(cookie: Integer; _type: _DATA_OBJECT_TYPES;
                           out ppDataObject: IDataObject): HResult; stdcall;
  function ComponentGetResultViewType(cookie: Integer; out ppViewType: PWideChar; out pViewOptions: Integer): HResult; stdcall;
  function ComponentGetDisplayInfo(var pResultDataItem: _RESULTDATAITEM): HResult; stdcall;
  function ComponentCompareObjects(const lpDataObjectA: IDataObject; const lpDataObjectB: IDataObject): HResult; stdcall;

  function ExtendContextMenuCommand (nCommandID : Integer; const dataObject : IDataObject) : HRESULT; stdcall;
  function ExtendContextMenuAddMenuItems (const dataObject : IDataObject;  const piCallback: IContextMenuCallback; var pInsertionAllowed: Integer) : HRESULT; stdcall;

  // IExtendControlbar
  function ExtendControlbarSetControlbar(const pControlbar: IControlbar): HResult; stdcall;
  function ExtendControlbarControlbarNotify(event: _MMC_NOTIFY_TYPE; arg, param: Integer): HResult; stdcall;

  // IResultDataCompare
  function ResultDataCompareCompare(lUserParam, cookieA, cookieB: Integer; var pnResult: SYSINT): HResult; stdcall;

  function ExtendPropertySheetCreatePropertyPages(const lpProvider: IPropertySheetCallback; handle: Integer;
                                 const lpIDataObject: IDataObject): HResult; stdcall;
  function ExtendPropertySheetQueryPagesFor(const lpDataObject: IDataObject): HResult; stdcall;

end;

TDataObject = class (TInterfacedObject, IDataObject)
private
  fInternal : TInternal;
    function CreateCoClassID(var medium: TStgMedium): HRESULT;
    function CreateData(buffer: pointer; len: Integer;
      var medium: TStgMedium): HRESULT;
    function CreateDisplayName(var medium: TStgMedium): HRESULT;
    function CreateInternal(var medium: TstgMedium): HRESULT;
    function CreateNodeTypeData(var medium: TStgMedium): HRESULT;
    function CreateszNodeType(var medium: TstgMedium): HRESULT;
    function CreateWorkstationName(var medium: TstgMedium): HRESULT;
protected
    function GetData(const formatetcIn: TFormatEtc; out medium: TStgMedium):
      HResult; stdcall;
    function GetDataHere(const formatetc: TFormatEtc; out medium: TStgMedium):
      HResult; stdcall;
    function QueryGetData(const formatetc: TFormatEtc): HResult;
      stdcall;
    function GetCanonicalFormatEtc(const formatetc: TFormatEtc;
      out formatetcOut: TFormatEtc): HResult; stdcall;
    function SetData(const formatetc: TFormatEtc; var medium: TStgMedium;
      fRelease: BOOL): HResult; stdcall;
    function EnumFormatEtc(dwDirection: Longint; out enumFormatEtc:
      IEnumFormatEtc): HResult; stdcall;
    function DAdvise(const formatetc: TFormatEtc; advf: Longint;
      const advSink: IAdviseSink; out dwConnection: Longint): HResult; stdcall;
    function DUnadvise(dwConnection: Longint): HResult; stdcall;
    function EnumDAdvise(out enumAdvise: IEnumStatData): HResult;
      stdcall;
end;

TComponentDataClass = class of TSnapinComponentData;
TSnapinAboutClass = class of TSnapinAbout;

implementation

const
                                          // IDataObject formats for snapin data
  s_cfDisplayName    : Integer = 0;
  s_cfNodeType       : Integer = 0;
  s_cfCoClass        : Integer = 0;
  s_cfInternal       : Integer = 0;
  s_cfMultiSel       : Integer = 0;
  s_cfszNodeType     : Integer = 0;
  s_cfWorkstation    : Integer = 0;

                                          // Dummy GUIDS : Our objects scope and
                                          // result objects aren't OLE objects
  cNodeTypeStatic       : TGuid = '{9E186021-3B8A-11D3-87F9-0001FA688BCB}';
  cObjectTypeResultItem : TGuid = '{9E186022-3B8A-11D3-87F9-0001FA688BCB}';
  cNodeTypeScope        : TGuid = '{9E186023-3B8A-11D3-87F9-0001FA688BCB}';

  /////////////////////////////////////////////////////////////////////////////
  // These should be in MMC_TLB, but the import did clever COM stuff - so
  // they're not...

                                          // MMC clipboard (IDataObject) formats
  CCF_SNAPIN_INTERNAL              = 'SNAPIN_INTERNAL';
  CCF_SNAPIN_WORKSTATION           = 'SNAPIN_WORKSTATION';
  CCF_NODETYPE                     = 'CCF_NODETYPE';
  CCF_NODEID                       = 'CCF_NODEID';
  CCF_SZNODETYPE                   = 'CCF_SZNODETYPE';
  CCF_DISPLAY_NAME                 = 'CCF_DISPLAY_NAME';
  CCF_SNAPIN_CLASSID               = 'CCF_SNAPIN_CLASSID';
  CCF_MMC_MULTISELECT_DATAOBJECT   = 'CCF_MMC_MULTISELECT_DATAOBJECT';
  CCF_MULTI_SELECT_SNAPINS         = 'CCF_MULTI_SELECT_SNAPINS';
  CCF_OBJECT_TYPES_IN_MULTI_SELECT = 'CCF_OBJECT_TYPES_IN_MULTI_SELECT';

                                          // MMC View enums
  MMC_VIEW_OPTIONS_NONE          = 0;
  MMC_VIEW_OPTIONS_NOLISTVIEWS   = $1;
  MMC_VIEW_OPTIONS_MULTISELECT   = $2;
  MMC_VIEW_OPTIONS_OWNERDATALIST = $4;
  MMC_VIEW_OPTIONS_FILTERED      = $8;
  MMC_VIEW_OPTIONS_CREATENEW     = $10;

                                         //  MMC_CONSOLE_VERB enum
  MMC_VERB_NONE       = 0;
  MMC_VERB_OPEN       = $8000;
  MMC_VERB_COPY       = $8001;
  MMC_VERB_PASTE      = $8002;
  MMC_VERB_DELETE     = $8003;
  MMC_VERB_PROPERTIES = $8004;
  MMC_VERB_RENAME     = $8005;
  MMC_VERB_REFRESH    = $8006;
  MMC_VERB_PRINT      = $8007;

                                           // MMC_NOTIFY_TYPE enum
  MMCN_ACTIVATE        = $8001;
  MMCN_ADD_IMAGES      = $8002;
  MMCN_BTN_CLICK       = $8003;
  MMCN_CLICK           = $8004;
  MMCN_COLUMN_CLICK    = $8005;
  MMCN_CONTEXTMENU     = $8006;
  MMCN_CUTORMOVE       = $8007;
  MMCN_DBLCLICK        = $8008;
  MMCN_DELETE          = $8009;
  MMCN_DESELECT_ALL    = $800a;
  MMCN_EXPAND          = $800b;
  MMCN_HELP            = $800c;
  MMCN_MENU_BTNCLICK   = $800d;
  MMCN_MINIMIZED       = $800e;
  MMCN_PASTE           = $800f;
  MMCN_PROPERTY_CHANGE = $8010;
  MMCN_QUERY_PASTE     = $8011;
  MMCN_REFRESH         = $8012;
  MMCN_REMOVE_CHILDREN = $8013;
  MMCN_RENAME          = $8014;
  MMCN_SELECT          = $8015;
  MMCN_SHOW            = $8016;
  MMCN_VIEW_CHANGE     = $8017;
  MMCN_SNAPINHELP      = $8018;
  MMCN_CONTEXTHELP     = $8019;
  MMCN_INITOCX         = $801a;
                                           // SCOPEDATAITEM constants
  SDI_STR        = $2;
  SDI_IMAGE      = $4;
  SDI_OPENIMAGE  = $8;
  SDI_STATE      = $10;
  SDI_PARAM      = $20;
  SDI_CHILDREN   = $40;
  SDI_PARENT     = 0;
  SDI_PREVIOUS   = $10000000;
  SDI_NEXT       = $20000000;
  SDI_FIRST      = $8000000;
  MMC_CALLBACK = PWideChar (-1);

  RDI_STR    = $2;                          // RESULTDATAITEM constants
  RDI_IMAGE  = $4;
  RDI_STATE  = $8;
  RDI_PARAM  = $10;
  RDI_INDEX  = $20;
  RDI_INDENT = $40;

(*--------------------------------------------------------------------------*
 | function HandleException : HResult                                       |
 |                                                                          |
 | Convert exceptions into HResult.  No snapin functions should raise       |
 | exceptions.                                                              |
 *--------------------------------------------------------------------------*)
function HandleException: HResult;
var
  E: TObject;
begin
  E := ExceptObject;
  if (E is EOleSysError) and (EOleSysError(E).ErrorCode < 0) then
    Result := EOleSysError(E).ErrorCode else
    Result := E_UNEXPECTED;
end;

(*--------------------------------------------------------------------------*
 | function IS_SPECIAL_DATAOBJECT ()                                        |
 |                                                                          |
 | The IS_SPECIAL_DATAOBJECT macro determines whether an LPDATAOBJECT       |
 | passed by MMC in a call to the snap-in's Notify method is a special type |
 | of data object instead of a pointer to an actual IDataObject object.     |
 |                                                                          |
 | Parameters:                                                              |
 |   d : IDataObject               The data object to test                  |
 *--------------------------------------------------------------------------*)
function IS_SPECIAL_DATAOBJECT (d : IDataObject) : BOOL;
begin
  result := (Integer (d) >= -10) and (Integer (d) <= 0)
end;

(*--------------------------------------------------------------------------*
 | function ExtractFromDataObject () : HRESULT                              |
 |                                                                          |
 | Extract data from a data object in the required format                   |
 |                                                                          |
 | Parameters:                                                              |
 |   dataObject : IDataObject             The data object to extract from   |
 |   cf : UINT                            The required clipboard format.    |
 |   cb : UINT                            Number of bytes to extract.       |
 |   var hglob : HGLOBAL                  Returns the data chunk.  This     |
 |                                        must be free'd with GlobalFree.   |
 |                                                                          |
 | The function returns an OLE success code.                                |
 *--------------------------------------------------------------------------*)
function ExtractFromDataObject (dataObject : IDataObject; cf : UINT; cb : LongInt; var hglob : HGLOBAL) : HRESULT;
var
  stgmedium : TStgMedium;
  formatetc : TFormatETC;
begin
  try
    result := S_FALSE;
    try
      stgmedium.tymed := TYMED_HGLOBAL;
      stgmedium.hBitmap := 0;

      formatetc.cfFormat := cf;
      formatetc.ptd := Nil;
      formatetc.dwAspect := DVASPECT_CONTENT;
      formatetc.lindex := -1;
      formatetc.tymed := TYMED_HGLOBAL;

      Assert (dataObject <> Nil);
      hglob := 0;

      stgmedium.hGlobal := GlobalAlloc (GMEM_SHARE, cb);
      if stgmedium.hGlobal = 0 then
        result := E_OUTOFMEMORY
      else
      begin
        result := dataObject.GetDataHere (formatetc, stgmedium);
        if result = S_OK then
        begin
          hglob := stgmedium.hGlobal;
          stgmedium.hGlobal := 0
        end
      end;

    finally
      if (result <> S_OK) and (stgmedium.hGlobal <> 0) then
        GlobalFree (stgmedium.hGlobal)
    end

  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function ExtractInternalFormat () : PInternal;                           |
 |                                                                          |
 | Extract internal format data from the data object.  The internal format  |
 | data can contain a scope item or a result item.                          |
 |                                                                          |
 | The pointer returned must be free'd with GlobalFree                      |
 |                                                                          |
 | Parameters:                                                              |
 |   dataObject : IDataObject      The data object to extract from.         |
 |                                                                          |
 | The function returns the internal format data.                           |
 *--------------------------------------------------------------------------*)
function ExtractInternalFormat (dataObject : IDataObject) : PInternal;
var
  data : HGLOBAL;
  rc : HRESULT;
begin
  rc := ExtractFromDataObject (dataObject, s_cfInternal, sizeof (TInternal), data);
  if rc = S_OK then
    result := PInternal (data)
  else
    result := Nil
end;

(*--------------------------------------------------------------------------*
 | function ImageListToBitmap () : HBITMAP;                                 |
 |                                                                          |
 | Convert an image list to a bitmap strip of images, each one 'size'       |
 | pixels wide and high.                                                    |
 |                                                                          |
 | Note that imageList.GetImageBitmap doesn't appear to work...             |
 |                                                                          |
 | Note that the bitmap returned must be free'd with DeleteObject           |
 |                                                                          |
 | Parameters:                                                              |
 |   imageList : TImageList       The image list to convert                 |
 |   size : Integer               The size of each image in the strip.      |
 |                                                                          |
 | The function returns the strip bitmap                                    |
 *--------------------------------------------------------------------------*)
function ImageListToBitmap (imageList : TImageList; size : Integer; var mask : UINT) : HBITMAP;
var
  bitmap, bmp : TBitmap;
  i : Integer;
  r : TRect;

const
  n : Integer = 0;

begin
  bmp := Nil;
  bitmap := TBitmap.Create;
  mask := RGB (123, 234, 235);
  try
    bmp := TBitmap.Create;
    bmp.Width := size;
    bmp.Height := size;
                                           // Create a bitmap to hold all the
                                           // images.
    bitmap.Width := size * imageList.Count;
    bitmap.Height := size;

    for i := 0 to imageList.count - 1 do   // Add each image to the bitmap
    begin
      bmp.Canvas.FillRect (rect (0, 0, bmp.width, bmp.height));
      imageList.GetBitmap (i, bmp);
      r.Left := i * size;
      r.Right := r.Left + size;
      r.Top := 0;
      r.Bottom := size;
      bitmap.Canvas.StretchDraw (r, bmp);
    end;

    result := bitmap.ReleaseHandle;

  finally
    bitmap.Free;
    bmp.Free
  end
end;

(*--------------------------------------------------------------------------*
 | procedure SetImageList ()                                                |
 |                                                                          |
 | Set an IImageList's bitmaps to contain the images from two image lists.  |
 | IImageList insists that it's bitmaps are 16x16 and 32x32, so we have to  |
 | enforce that, converting images ti the correct size if possible.         |
 |                                                                          |
 | Parameters:                                                              |
 |   imageList : IImageList       The image list to fill                    |
 |   smallImageList,                                                        |
 |   largeImageLis : TImageList   The image lists to get the images from.   |
 *--------------------------------------------------------------------------*)
procedure SetImageList (imageList : IImageList; smallImageList, largeImageList : TImageList);
var
  smallImages, largeImages : HBITMAP;
  mask : UINT;
begin
  if Assigned (smallImageList) or Assigned (largeImageList) then
  begin
    smallImages := 0;
    largeImages := 0;
    try
      if Assigned (smallImageList) then
        smallImages := ImageListToBitmap (smallImageList, 16, mask)
      else
        smallImages := ImageListToBitmap (largeImageList, 16, mask);

      if Assigned (largeImageList) then
        largeImages := ImageListToBitmap (largeImageList, 32, mask)
      else
        largeImages := ImageListToBitmap (smallImageList, 32, mask);


      OleCheck (imageList.ImageListSetStrip (smallImages, largeImages, 0, mask));
    finally
      if (smallImages <> 0) then
        DeleteObject (smallImages);
      if (largeImages <> 0) then
        DeleteObject (largeImages);
    end
  end
end;

procedure GetContextMenuFlags (itm : TMenuItem; var flags, specialFlags : Integer);
begin
  flags := 0;
  specialFlags := 0;

  if itm.Enabled then
    flags := flags or MF_ENABLED
  else
    flags := flags or MF_DISABLED or MF_GRAYED;

  if itm.Checked then
    flags := flags or MF_CHECKED
  else
    flags := flags or MF_UNCHECKED;

  if itm.Default then
    specialFlags := specialFlags or CCM_SPECIAL_DEFAULT_ITEM;
end;

{ TSnapinComponentData }


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.ComponentDataCompareObjects ()             |
 |                                                                          |
 |                                                                          |
 |                                                                          |
 | Parameters:                                                              |
 |   lpDataObjectA,                                                         |
 |   lpDataObjectB  : IDataObject           The objects to compare.         |
 |                                                                          |
 | The function returns S_OK if the objects match, or S_FALSE if they don't |
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ComponentDataCompareObjects(const lpDataObjectA,
  lpDataObjectB: IDataObject): HResult;
var
  objA, objB : PInternal;
begin
  try
    result := S_FALSE;

    objA := ExtractInternalFormat (lpDataObjectA);
    if Assigned (objA) then
    try
      objB := ExtractInternalFormat (lpDataObjectB);

      if Assigned (objB) then
      try
        if (ObjA^.m_type = ObjB^.m_type) and (ObjA^.m_Cookie = ObjB^.m_Cookie) then
          result := S_OK;
      finally
        GlobalFree (THandle (ObjB))
      end
    finally
      GlobalFree (THandle (ObjA))
    end
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.ComponentDataCreateComponent ()            |
 |                                                                          |
 | Create a 'result' pane component for the component data.                 |
 |                                                                          |
 | Parameters:                                                              |
 |   out ppComponent : IComponent       The created component.              |
 |                                                                          |
 | The function returns an OLE success code                                 |
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ComponentDataCreateComponent(
  out ppComponent: IComponent): HResult;
var
  Component : TSnapinComponent;
begin
  try
    Component := TSnapinComponent.Create;
    Component.fParent := self;
    ppComponent := Component;
    result := S_OK;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.ComponentDataDestroy.                      |
 |                                                                          |
 | Destroy the component data.  Release the interfaces obtained in          |
 | Initialize.                                                              |
 |                                                                          |
 | The function returns an OLE success code                                 |
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ComponentDataDestroy: HResult;
begin
  try
    fScopeImageList := Nil;
    fConsole := Nil;
    fConsole2 := Nil;
    fConsoleNameSpace := Nil;
    result := S_OK;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.ComponentDataGetDisplayInfo                |
 |                                                                          |
 | Return display information - The only thing used is the 'text' - which   |
 | appears in the scope pane tree view root for the scope data item.        |
 |                                                                          |
 | Parameters:                                                              |
 |   var pScopeDataItem : _SCOPEDATAITEM  On entry, lParam contains the     |
 |                        scope node, and 'mask' contains flags for the     |
 |                        in formation required.                            |
 |                                                                          |
 | The function returns an OLE success code                                 |
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ComponentDataGetDisplayInfo(
  var pScopeDataItem: _SCOPEDATAITEM): HResult;
var
  node : TScopeItem;
begin
  try
    node := TScopeItem (pScopeDataItem.lParam);
    if (pScopeDataItem.mask and SDI_STR) <> 0 then
      pScopeDataItem.displayName := PWideChar (WideString (node.Text));

    result := S_OK
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.ComponentDataInitialize                    |
 |                                                                          |
 | Initialize the component data by obtaining the IConsole and associated   |
 | interfaces.                                                              |
 |                                                                          |
 | Parameters:                                                              |
 |   pUnknown : IUnknown          Unknown interface that must inherit from  |
 |                                IConsole.                                 |
 |                                                                          |
 | The function returns an OLE success code                                 |
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ComponentDataInitialize(
  const pUnknown: IUnknown): HResult;
begin
  FIsDirty := True;
  try
    fConsole := pUnknown as IConsole;
    try
      fConsole2 := fConsole as IConsole2;
    except
      fConsole2 := Nil
    end;
    fConsoleNameSpace := pUnknown as IConsoleNameSpace;
    fConsole.QueryScopeImageList (fScopeImageList);

    FInitialized := True;
    SetScopeImages;                     // Set the scope pane images.


    result := S_OK;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.ComponentDataNotify                        |
 |                                                                          |
 | Handle ComponentData (Scope Pane) notifications.                         |
 |                                                                          |
 | Parameters:                                                              |
 |   lpDataObject : IDataObject         Either a 'special data object' or   |
 |                                      a scope item object.                |
 |   event         : _MMC_NOTIFY_TYPE   The nofication                      |
 |   arg           : Integer            Notification  argument.             |
 |   param         : Integer            Additional notifcation parameter.   |
 |                                                                          |
 | The function returns an OLE success code                                 |
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ComponentDataNotify(
  const lpDataObject: IDataObject; event: _MMC_NOTIFY_TYPE; arg,
  param: Integer): HResult;
var
  scopeItem : TScopeItem;

begin
  try
    result := S_OK;
    Assert (fConsoleNameSpace <> Nil);


    if event = MMCN_PROPERTY_CHANGE then
      MessageBox (0, 'Property Change', '', MB_ICONINFORMATION)
    else
    begin
      scopeItem := GetNodeForDataObject (lpDataObject) as TScopeItem;

      if Assigned (scopeItem) then
      case event of
        MMCN_CONTEXTMENU     : MessageBox (0, 'TTTestSnapin.MMCN_HELP', Nil, MB_ICONINFORMATION);
        MMCN_CONTEXTHELP     : MessageBox (0, 'TTTestSnapin.MMCN_HELP', Nil, MB_ICONINFORMATION);
        MMCN_PASTE           : MessageBox (0, 'TTTestSnapin.MMCN_PASTE', Nil, MB_ICONINFORMATION);
        MMCN_REMOVE_CHILDREN : ;
        MMCN_COLUMN_CLICK    : MessageBox (0, 'ColumnClick', Nil, MB_ICONINFORMATION);
        MMCN_EXPAND          :

          begin              // The most important notification.
                             // Enumerate the scope pane to return the child nodes


            if BOOL (arg) = TRUE then
            begin
              if Assigned (scopeItem.SnapinData.OnScopeExpand) then
                scopeItem.SnapinData.OnScopeExpand (scopeItem);
              EnumerateScopePane (lpDataObject, param)
            end
          end;
      end
    end
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.ComponentDataQueryDataObject               |
 |                                                                          |
 | Create a data object for this ComponentDataObject.                       |
 |                                                                          |
 | Parameters:                                                              |
 |   cookie : Integer           The cookie to associate with the data       |
 |   _type  : DATA_OBJECT_TYPE  Whether it's a scope or result data object  |
 |   out ppDataObject : IDataObject    Returns the new data object.         |
 |                                                                          |
 | The function returns an OLE success code                                 |
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ComponentDataQueryDataObject(cookie: Integer;
  _type: _DATA_OBJECT_TYPES; out ppDataObject: IDataObject): HResult;
var
  dataObject : TDataObject;
begin
  try
    dataObject := TDataObject.Create;
    dataObject.fInternal.m_type := _type;
    dataObject.fInternal.m_cookie := cookie;
    dataObject.fInternal.m_clsid := Factory.ClassID;
    dataObject.fInternal.m_object := self;          // fInternal points to 'self'
                                                    // so that we can retrieve
                                                    // ourself from data objects
    ppDataObject := dataObject as IDataObject;
    result := S_OK;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.EnumerateScopePane                         |
 |                                                                          |
 | dataObject : IDataObject                                                 |
 |                                                                          |
 | Parameters:                                                              |
 |   dataObject : IDataObject.  Contains the scope item to enumerate        |
 |   pParent    : HSCOPEITEM    The scope item's parent node handle.        |
 |                                                                          |
 | The function returns an OLE success code                                 |
 *--------------------------------------------------------------------------*)
procedure TSnapinComponentData.EnumerateScopePane(dataObject: IDataObject;
  pParent: HSCOPEITEM);
var
  i : Integer;
  item : _SCOPEDATAITEM;
  hr : HRESULT;
  root, node : TScopeItem;

begin
  root := GetNodeForDataObject (dataObject) as TScopeItem;

  if Assigned (root) then
  begin
                         // Insert the scope items.

    for i := 0 to root.ScopeItems.Count - 1 do
    begin
      node := root.ScopeItems [i];
      FillChar (item, sizeof (item), 0);
      item.mask := SDI_STR or SDI_PARAM or SDI_PARENT or SDI_CHILDREN;

      if node.ImageIndex > -1 then      // Set the image
      begin
        item.mask := item.mask or SDI_IMAGE or SDI_OPENIMAGE;
        item.nImage := node.ImageIndex;
        item.nOpenImage := node.ImageIndex;
      end;

      item.cChildren := node.ScopeItems.Count;
      item.relativeID := pParent;
      item.displayname := MMC_CALLBACK;
      item.lParam := Integer (node);
      hr := FConsoleNameSpace.InsertItem (item);
      OleCheck (hr);
    end
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ExtendContextMenuAddMenuItems(
  const dataObject: IDataObject; const piCallback: IContextMenuCallback;
  var pInsertionAllowed: Integer): HRESULT;
var
  scopeItem : TScopeItem;
  i : Integer;
  item : _CONTEXTMENUITEM;
  itm : TMenuItem;
begin
  try
    result := S_OK;
    scopeItem := GetNodeForDataObject (dataObject) as TScopeItem;
    if Assigned (scopeItem) and Assigned (scopeItem.ContextMenu) then
    begin
      for i := 0 to scopeItem.ContextMenu.Items.Count - 1 do
      begin
        FillChar (item, sizeof (item), 0);
        itm := scopeItem.ContextMenu.Items [i];
        item.strName := PWideChar (WideString (itm.Caption));
        item.strStatusBarText := PWideChar (WideString (itm.Hint));
        item.lCommandID := itm.Command;

        GetContextMenuFlags (itm, item.fFlags, item.fSpecialFlags);
        item.lInsertionPointID := Integer (CCM_INSERTIONPOINTID_PRIMARY_TOP);

        result := piCallback.AddItem (item);
        if not Succeeded (result) then
          break;
      end
    end
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ExtendContextMenuCommand(nCommandID: Integer;
  const dataObject: IDataObject): HRESULT;
var
  node : TScopeItem;
begin
  try
    node := GetNodeForDataObject (dataObject) as TScopeItem;
    if Assigned (node) and Assigned (node.ContextMenu) then
      node.ContextMenu.DispatchCommand (nCommandID);

    result := S_OK;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ExtendPropertySheetCreatePropertyPages(
  const lpProvider: IPropertySheetCallback; handle: Integer;
  const lpIDataObject: IDataObject): HResult;
begin
  try
    result := S_FALSE;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.ExtendPropertySheetQueryPagesFor(
  const lpDataObject: IDataObject): HResult;
begin
  try
    result := S_FALSE;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.GetNodeForCookie(
  cookie: Integer): TObject;
begin
  if cookie = 0 then
    result := FSnapinData.ScopeItem
  else
    result := TObject (cookie);
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.GetNodeForDataObject(
  const dataObj: IDataObject): TObject;
var
  internal : PInternal;
begin
  Internal := ExtractInternalFormat (dataObj);
  result := Nil;
  if Assigned (Internal) then
  try
    result := GetNodeForCookie (Internal^.m_cookie);
  finally
    GlobalFree (Integer (internal))
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.GetWindowHandle: HWND;
begin
  if Assigned (fConsole) then
    fConsole.GetMainWindow (wireHWND (result))
  else
    result := 0;
end;

procedure TSnapinComponentData.Initialize;
begin
  fSnapinData := GetSnapinData;
  fSnapinData.Parent := self;
  s_cfInternal       := RegisterClipboardFormat (CCF_SNAPIN_INTERNAL);
  s_cfDisplayName    := RegisterClipboardFormat (CCF_DISPLAY_NAME);
  s_cfNodeType       := RegisterClipboardFormat (CCF_NODETYPE);
  s_cfCoClass        := RegisterClipboardFormat (CCF_SNAPIN_CLASSID);
  s_cfMultiSel       := RegisterClipboardFormat (CCF_OBJECT_TYPES_IN_MULTI_SELECT);
  s_cfWorkstation    := RegisterClipboardFormat (CCF_SNAPIN_WORKSTATION);
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.PersistStreamGetClassID(
  out classID: TCLSID): HResult;
begin
  try
    classID := Factory.ClassID;

    result := S_OK;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.PersistStreamGetSizeMax(
  out cbSize: Largeint): HResult;
begin
  try
    cbSize := 10;
    result := S_OK;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.PersistStreamIsDirty: HResult;
begin
  try
    if fIsDirty then
      result := S_OK
    else
      result := S_FALSE;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.PersistStreamLoad(
  const stm: IStream): HResult;
var
  psz : array [0..9] of char;
  nBytesRead : Longint;
begin
  try
    Assert (fInitialized);

    result := stm.Read (@psz [0], 10, @nBytesRead);
    Assert (Succeeded (result) and (nBytesRead = 10));
    Assert (lstrcmp (psz, '987654321') = 0);
    fIsDirty := False;
    if Succeeded (result) then
      result := S_OK
    else
      result := E_FAIL
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponentData.PersistStreamSave(const stm: IStream;
  fClearDirty: BOOL): HResult;
var
  nBytesWritten : Longint;
begin
  try
    Assert (fInitialized);
    result := stm.Write (PChar ('987654321'), 10, @nBytesWritten);

    Assert (Succeeded (result) and (nBytesWritten = 10));
    if Failed (result) then
      result := STG_E_CANTSAVE
    else
    begin
      fIsDirty := False;
      result := S_OK
    end
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponentData.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
procedure TSnapinComponentData.SetScopeImages;
begin
  if Assigned (fSnapinData) and Initialized then
    SetImageList (fScopeImageList, fSnapinData.ScopeSmallImages, fSnapinData.ScopeLargeImages);
end;

{ TDataObject }


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.CreateData(buffer: pointer; len: Integer;
  var medium: TStgMedium): HRESULT;
var
  stream : IStream;
  written : LongInt;
begin
  try
    if (not Assigned (buffer)) then
       result := E_POINTER
    else
    begin
      result := DV_E_TYMED;

      if medium.tymed = TYMED_HGLOBAL then
      begin
        result := CreateStreamOnHGlobal (medium.hGlobal, FALSE, stream);

        if result = S_OK then
        begin
          result := stream.Write (buffer, len, @written);
        end
      end
    end
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.CreateCoClassID (var medium : TStgMedium) : HRESULT;
begin
  result :=CreateData (@fInternal.m_clsid, sizeof (TGuid), medium)
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.CreateDisplayName(var medium: TStgMedium): HRESULT;
var
  szDispName : WideString;
  componentData : TSnapinComponentData;
begin
  componentData := TSnapinComponentData (fInternal.m_Object);
  szDispName := componentData.fSnapinData.ScopeItem.Text;
  result := CreateData (PWideChar (szDispName), (Length (szDispName) + 1) * sizeof (WideChar), medium);
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.CreateInternal (var medium : TstgMedium) : HRESULT;
begin
  result := CreateData (@fInternal, sizeof (fInternal), medium);
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.CreateNodeTypeData(var medium: TStgMedium): HRESULT;
var
  pcObjectType : PGuid;
begin
  result := S_OK;
  pcObjectType := Nil;
  if fInternal.m_cookie = 0 then
    pcObjectType := @cNodeTypeStatic
  else
    if fInternal.m_type = CCT_SCOPE then
      pcObjectType := @cNodeTypeScope
    else
      if fInternal.m_type = CCT_RESULT then
        pcObjectType := @cObjectTypeResultItem;

  if result = S_OK then
    result := CreateData (pcObjectType, sizeof (TGuid), medium);
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.DAdvise(const formatetc: TFormatEtc; advf: Integer;
  const advSink: IAdviseSink; out dwConnection: Integer): HResult;
begin
  result := E_NOTIMPL;
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.DUnadvise(dwConnection: Integer): HResult;
begin
  result := E_NOTIMPL;
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.EnumDAdvise(out enumAdvise: IEnumStatData): HResult;
begin
  result := E_NOTIMPL;
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.EnumFormatEtc(dwDirection: Integer;
  out enumFormatEtc: IEnumFormatEtc): HResult;
begin
  result := E_NOTIMPL;
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.GetCanonicalFormatEtc(const formatetc: TFormatEtc;
  out formatetcOut: TFormatEtc): HResult;
begin
  result := E_NOTIMPL;
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.GetData(const formatetcIn: TFormatEtc;
  out medium: TStgMedium): HResult;
begin
  result := E_NOTIMPL;
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.GetDataHere(const formatetc: TFormatEtc;
  out medium: TStgMedium): HResult;
var
  cf : word;
begin
  try
    result := DV_E_CLIPFORMAT;
    cf := formatetc.cfFormat;

    if cf = s_cfNodeType then
      result := CreateNodeTypeData (medium)
    else
      if cf = s_cfCoClass then
        result := CreateCoClassID (medium)
      else
        if cf = s_cfszNodeType then
          result := CreateszNodeType (medium)
        else
          if cf = s_cfDisplayName then
            result := CreateDisplayName (medium)
          else
            if cf = s_cfInternal then
              result := CreateInternal (medium)
            else
              if cf = s_cfWorkstation then
                result := CreateWorkstationName (medium)
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.QueryGetData(const formatetc: TFormatEtc): HResult;
begin
  result := E_NOTIMPL;
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.SetData(const formatetc: TFormatEtc;
  var medium: TStgMedium; fRelease: BOOL): HResult;
begin
  result := E_NOTIMPL;
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.CreateszNodeType(var medium: TstgMedium): HRESULT;
var
  wszNodeType : WideString;
  pcObjectType : PGUID;
begin
  try
    pcObjectType := Nil;
    if fInternal.m_cookie = 0 then
      pcObjectType := @cNodeTypeStatic
    else
      if fInternal.m_type = CCT_SCOPE then
        pcObjectType := @cNodeTypeStatic
      else
        if fInternal.m_type = CCT_RESULT then
          pcObjectType := @cObjectTypeResultItem;

    if Assigned (pcObjectType) then
    begin
      wszNodeType := GUIDToString (pcObjectType^);
      result := CreateData (PWideChar (wszNodeType), (Length (wszNodeType) + 1) * sizeof (WideChar), medium)
    end
    else
      result := E_FAIL
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TDataObject.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TDataObject.CreateWorkstationName(
  var medium: TstgMedium): HRESULT;
var
  buffer : array [0..MAX_COMPUTERNAME_LENGTH] of char;
  bufLen : DWORD;
  wComputerName : WideString;
begin
  try
    bufLen := sizeof (buffer);
    if GetComputerName (buffer, bufLen) then
    begin
      wComputerName := buffer;
      result := CreateData (PWideChar (wComputerName), (Length (wComputerName) + 1) * sizeof (WideChar), medium);
    end
    else
      result := E_FAIL
  except
    result := HandleException
  end
end;

{ TSnapinComponent }

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ComponentCompareObjects(const lpDataObjectA,
  lpDataObjectB: IDataObject): HResult;
var
  objA, objB : PInternal;
begin
  try
    result := S_FALSE;

    objA := ExtractInternalFormat (lpDataObjectA);
    if Assigned (objA) then
    try
      objB := ExtractInternalFormat (lpDataObjectB);

      if Assigned (objB) then
      try
        if (ObjA^.m_type = ObjB^.m_type) and (ObjA^.m_Cookie = ObjB^.m_Cookie) then
          result := S_OK;
      finally
        GlobalFree (THandle (ObjB))
      end
    finally
      GlobalFree (THandle (ObjA))
    end
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ComponentDestroy(cookie: Integer): HResult;
begin
  try
    fConsole2 := Nil;
    fResultData := Nil;
    fResultImages := Nil;
    fConsole := Nil;
    fHeader := Nil;
    fConsoleVerb := Nil;
    result := S_OK;
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ComponentGetDisplayInfo(
  var pResultDataItem: _RESULTDATAITEM): HResult;
var
  item : TScopeItem;
  resultItem : TResultItem;
begin
  try
    if pResultDataItem.bScopeItem <> 0 then
    begin
      item := TObject (pResultDataItem.lParam) as TScopeItem;

      if (pResultDataItem.mask and RDI_STR) <> 0 then
        pResultDataItem.str := PWideChar (WideString (item.Text));

      if (pResultDataItem.mask and RDI_IMAGE) <> 0 then
        pResultDataItem.nImage := item.ImageIndex
    end
    else
    begin
      resultItem := TObject (pResultDataItem.lParam) as TResultItem;

      with pResultDataItem do
      begin
        if (mask and RDI_STR) <> 0 then
          if nCol = 0 then
            str := PWideChar (WideString (resultItem.Text))
          else
            if nCol <= resultItem.SubItems.Count then
              str := PWideChar (WideString (resultItem.SubItems [nCol - 1]));

        pResultDataItem.nImage := resultItem.ImageIndex;
      end
    end;

    result := S_OK
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ComponentGetResultViewType(cookie: Integer;
  out ppViewType: PWideChar; out pViewOptions: Integer): HResult;
var
  ScopeItem : TScopeItem;
begin
  try
    ScopeItem := fParent.GetNodeForCookie (cookie) as TScopeItem;

    if Assigned (ScopeItem) then
      case ScopeItem.ViewType of
        vtListView:
          begin
            ppViewType := Nil;
            pViewOptions := MMC_VIEW_OPTIONS_MULTISELECT
          end;

        vtGUID :
          begin
            ppViewType := StringToLPOleStr (ScopeItem.ViewTypeGUID);
            pViewOptions := MMC_VIEW_OPTIONS_NOLISTVIEWS
          end;

        vtHTML :
          begin
            ppViewType := StringToLPOleStr (ScopeItem.ViewTypeHTML);
            pViewOptions := MMC_VIEW_OPTIONS_NOLISTVIEWS
          end
      end;
    result := S_OK;
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ComponentInitialize(
  const lpConsole: IConsole): HResult;
begin
  try
    fConsole := lpConsole as IConsole;
    try
      fConsole2 := fConsole as IConsole2;
    except
      fConsole2 := Nil
    end;
    fResultData := fConsole as IResultData;
    try
      fHeader := fConsole as IHeaderCtrl;
      fConsole.SetHeader (fHeader);
    except
      fHeader := Nil
    end;
    fConsole.QueryConsoleVerb (fConsoleVerb);
    fConsole.QueryResultImageList (fResultImages);

    result := S_OK;
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ComponentNotify(const lpDataObject: IDataObject;
  event: _MMC_NOTIFY_TYPE; arg, param: Integer): HResult;
var
  resultItem : TResultItem;
  scopeItem : TScopeItem;
  allow : boolean;
  iid : Integer;
begin
  try
    result := S_OK;

    if IS_SPECIAL_DATAOBJECT (lpDataObject) then
    begin
      case event of
        MMCN_SNAPINHELP      : fParent.fSnapinData.HelpCommand;
        MMCN_PROPERTY_CHANGE :
          fResultData.UpdateItem (param);
      end
    end
    else
    begin
      if not fDeletingItem then
        DecodeDataObject (lpDataObject, scopeItem, resultItem)
      else
      begin
        scopeItem := Nil;
        resultItem := Nil
      end;

      case event of
        MMCN_VIEW_CHANGE :
            if Assigned (resultItem) then
              fResultData.UpdateItem (resultItem.itemID)
            else
              if Assigned (scopeItem) and (scopeItem.ViewType = vtListView) then
              begin
                if not fInShowing and (scopeItem = fCurrentScopeItem) then
                  if arg <> 0 then
                  begin
                    EnumerateResultPane (scopeItem);
                    SetResultImages (false);
                  end
                  else
                    fResultData.DeleteAllRsltItems
              end;

        MMCN_SHOW:
          begin
            if Assigned (scopeItem) then
            begin
              if Assigned (scopeItem.SnapinData.OnScopeShow) then
              begin
                fInShowing := True;
                try
                  scopeItem.SnapinData.OnScopeShow (scopeItem, arg <> 0);
                finally
                  fInShowing := False
                end
              end;

              if scopeItem.ViewType = vtListView then
              begin
                fCurrentScopeItem := scopeItem;
                if arg <> 0 then
                begin
                  InitializeHeaders (scopeItem);
                  EnumerateResultPane (scopeItem);
                  SetResultImages (scopeItem.scopeItems.Count > 0);
                end
                else
                  fResultData.DeleteAllRsltItems
              end
            end
          end;

        MMCN_CONTEXTHELP:
          scopeItem.HelpCommand;

        MMCN_CONTEXTMENU     : MessageBox (0, 'TTTestSnapin.MMCN_HELP', Nil, MB_ICONINFORMATION);
        MMCN_DESELECT_ALL,
        MMCN_SELECT       : HandleStandardVerbs(event = MMCN_DESELECT_ALL, arg, scopeItem, resultItem);
        MMCN_DBLCLICK :
          if Assigned (resultItem) and Assigned (ScopeItem.SnapinData.OnResultDblClick) then
            ScopeItem.SnapinData.OnResultDblClick (resultItem);

        MMCN_DELETE :
          if Assigned (resultItem) and Assigned (ScopeItem.SnapinData.OnResultDelete) then
          begin
            allow := True;
            iid := resultItem.itemID;
            ScopeItem.SnapinData.OnResultDelete (resultItem, allow);
            if allow then
            begin
              fDeletingItem := True;
              try
                fResultData.DeleteItem (iid, 0);
              finally
                fDeletingItem := False
              end
            end
          end;

        MMCN_RENAME :
          if Assigned (resultItem) and Assigned (ScopeItem.SnapinData.OnResultRename) then
          begin
            allow := True;
            ScopeItem.SnapinData.OnResultRename (resultItem, POleStr (param), allow);
            if not allow then result := S_FALSE;
          end;
      end
    end
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ComponentQueryDataObject(cookie: Integer;
  _type: _DATA_OBJECT_TYPES; out ppDataObject: IDataObject): HResult;
var
  dataObject : TDataObject;
begin
  try
    dataObject := TDataObject.Create;
    dataObject.fInternal.m_type := _type;
    dataObject.fInternal.m_cookie := cookie;
    dataObject.fInternal.m_clsid := fParent.Factory.ClassID;
    dataObject.fInternal.m_object := self;
    ppDataObject := dataObject as IDataObject;
    result := S_OK;
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
procedure TSnapinComponent.DecodeDataObject(dataObject: IDataObject;
  var scopeItem: TScopeItem; var resultItem: TResultItem);
var
  node : TObject;
begin
  node := fParent.GetNodeForDataObject (dataObject);
  resultItem := Nil;
  scopeItem := Nil;

  if Assigned (node) then
    if node is TScopeItem then
      scopeItem := TScopeItem (node)
    else
    begin
      resultItem := node as TResultItem;
      scopeItem := resultItem.ScopeItem
    end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
procedure TSnapinComponent.EnumerateResultPane(node: TScopeItem);
var
  i : Integer;
  resultItem : _RESULTDATAITEM;
begin
  with node do
  begin
    fResultData.SetItemCount (ResultItems.Count, 0);
    for i := 0 to ResultItems.Count - 1 do
    begin
      FillChar (resultItem, sizeof (resultItem), 0);
      resultItem.mask := RDI_STR or RDI_IMAGE or RDI_PARAM or RDI_INDEX;
      resultItem.str := MMC_CALLBACK;
      resultItem.nImage := node.ResultItems [i].ImageIndex;
      resultItem.lParam := Integer (node.ResultItems [i]);
      fResultData.InsertItem (resultItem);
      node.ResultItems [i].itemID := resultItem.itemID;
    end;

    fResultData.Sort (0, 0, -1);
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ExtendContextMenuAddMenuItems(
  const dataObject: IDataObject; const piCallback: IContextMenuCallback;
  var pInsertionAllowed: Integer): HRESULT;
var
  scopeItem : TScopeItem;
  resultItem : TResultItem;
  i : Integer;
  item : _CONTEXTMENUITEM;
  itm : TMenuItem;
begin
  try
    result := S_OK;
    DecodeDataObject (dataObject, scopeItem, resultItem);
    if Assigned (scopeItem) and Assigned (ScopeItem.ResultItemsContextMenu) then
    begin
      for i := 0 to ScopeItem.ResultItemsContextMenu.Items.Count - 1 do
      begin
        FillChar (item, sizeof (item), 0);
        itm := ScopeItem.ResultItemsContextMenu.Items [i];
        item.strName := PWideChar (WideString (itm.Caption));
        item.strStatusBarText := PWideChar (WideString (itm.Hint));
        item.lCommandID := itm.Command;
        GetContextMenuFlags (itm, item.fFlags, item.fSpecialFlags);
        item.lInsertionPointID := Integer (CCM_INSERTIONPOINTID_PRIMARY_TOP);

        result := piCallback.AddItem (item);
        if not Succeeded (result) then
          break;
      end
    end
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ExtendContextMenuCommand(nCommandID: Integer;
  const dataObject: IDataObject): HRESULT;
var
  node : TResultItem;
  scope : TScopeItem;
begin
  try
    DecodeDataObject (dataObject, scope, node);
    if Assigned (scope) and Assigned (scope.ResultItemsContextMenu) then
      scope.ResultItemsContextMenu.DispatchCommand (nCommandID);

    result := S_OK;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ExtendControlbarControlbarNotify(
  event: _MMC_NOTIFY_TYPE; arg, param: Integer): HResult;
begin
  try
    case event of
      MMCN_BTN_CLICK     :;
      MMCN_DESELECT_ALL  :;
      MMCN_SELECT        :;
      MMCN_MENU_BTNCLICK :;
    end;
    result := S_OK;
  except
    result := HandleException
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ExtendControlbarSetControlbar(
  const pControlbar: IControlbar): HResult;
begin
  result := S_FALSE;
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ExtendPropertySheetCreatePropertyPages(
  const lpProvider: IPropertySheetCallback; handle: Integer;
  const lpIDataObject: IDataObject): HResult;
var
  node : TResultItem;
  scope : TScopeItem;
  changed : boolean;
begin
  DecodeDataObject (lpIdataObject, scope, node);
  if Assigned (node) and Assigned (scope.SnapinData.OnResultProperties) then
  begin
    changed := False;
    scope.SnapinData.OnResultProperties (node, changed);
    if changed then
      fResultData.UpdateItem (node.itemID);

    result := S_FALSE;   // return false, otherwise MMC displays an empty propert sheet...
  end
  else
    result := S_FALSE
end;

function TSnapinComponent.ExtendPropertySheetQueryPagesFor(
  const lpDataObject: IDataObject): HResult;
var
  node : TResultItem;
  scope : TScopeItem;
begin
  DecodeDataObject (lpdataObject, scope, node);
  if Assigned (node) and Assigned (scope.SnapinData.OnResultProperties) then
    result := S_OK
  else
    result := S_FALSE;
end;

procedure TSnapinComponent.HandleStandardVerbs(bDeselectAll: boolean;
  arg: Integer; scopeItem : TScopeItem; resultItem : TResultItem);
var
  bScope, bSelect : boolean;

  procedure ShowResultOption (option: Integer; show : boolean);
  var
    iHide, iEnable : Integer;
  begin
    if show then
    begin
      iHide := 0;
      iEnable := 1
    end
    else
    begin
      iHide := 1;
      iEnable := 0
    end;
    fConsoleVerb.SetVerbState(option, HIDDEN, iHide);
    fConsoleVerb.SetVerbState(option, ENABLED, iEnable)
  end;

  procedure ShowResultOptions (show : boolean);
  begin
    ShowResultOption (MMC_VERB_DELETE, show and Assigned (scopeItem.SnapinData.OnResultDelete));
    ShowResultOption (MMC_VERB_PROPERTIES, show and Assigned (scopeItem.SnapinData.OnResultProperties));
    ShowResultOption (MMC_VERB_RENAME, show and Assigned (scopeItem.SnapinData.OnResultRename));
  end;

begin
  if scopeItem.ViewType <> vtListView then
  begin
    ShowResultOptions (False);
  end
  else
  begin
 { TODO : Handle Standard Verbs }
    bScope := WORDBOOL (LOWORD (arg));
    bSelect := WORDBOOL (HIWORD (arg));

    if bDeselectAll or not bSelect then
      ShowResultOptions (False)
    else
      ShowResultOptions (Assigned (resultItem));
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
procedure TSnapinComponent.InitializeHeaders(node: TScopeItem);
var
  i : Integer;
  column : TSnapinColumn;

  caption : WideString;
  align : Integer;
begin
  if Assigned (fHeader) then
  begin
    for i := 0 to node.Columns.Count - 1 do
    begin
      column := node.Columns [i];
      caption := column.Caption;
      case column.Alignment of
        taCenter : align := LVCFMT_CENTER;
        taRightJustify : align := LVCFMT_RIGHT;
        else align := LVCFMT_LEFT
      end;

      fHeader.InsertColumn (i, PWideChar (caption), align, column.Width)
    end
  end
end;


(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinComponent.ResultDataCompareCompare(lUserParam, cookieA,
  cookieB: Integer; var pnResult: SYSINT): HResult;
var
  nCol : Integer;
  szStringA, szStringB : string;
  pDataA, pDataB : TResultItem;
begin
  try
    nCol := pnResult;
    Assert ((nCol >= 0) and (nCol < 3));
    pnResult := 0;

    pDataA := TResultItem (cookieA);
    pDataB := TResultItem (cookieB);

    Assert (Assigned (pDataA) and Assigned (pDataB));

    if nCol = 0 then
    begin
      szStringA := pDataA.Text;
      szStringB := pDataB.Text;
    end
    else
    begin
      Dec (nCol);
      if nCol < pDataA.SubItems.Count then
        szStringA := pDataA.SubItems [nCol]
      else
        szStringA := '';

      if nCol < pDataB.SubItems.Count then
        szStringB := pDataB.SubItems [nCol]
      else
        szStringB := ''
    end;

    if szStringA < szStringB then
      pnResult := -1
    else
      if szStringA > szStringB then
        pnResult := 1
      else
        pnResult := 0;

    result := S_OK;

  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinComponent.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
procedure TSnapinComponent.SetResultImages (useScopeImages : boolean);
begin
  if Assigned (fParent.fSnapinData) and fParent.Initialized then
    if useScopeImages then
      SetImageList (fResultImages, fParent.fSnapinData.ScopeSmallImages, fParent.fSnapinData.ScopeLargeImages)
    else
      SetImageList (fResultImages, fParent.fSnapinData.ResultSmallImages, fParent.fSnapinData.ResultLargeImages);
end;

(*--------------------------------------------------------------------------*
 | function TSnapinAbout.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinAbout.GetProvider(out lpName: PWideChar): HResult;
begin
  try
    LoadSnapinData;
    lpName := StringToLPOleStr (fSnapinData.Provider);
    result := S_OK;
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinAbout.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinAbout.GetSnapinDescription(
  out lpDescription: PWideChar): HResult;
begin
  try
    LoadSnapinData;
    lpDescription := StringToLPOleStr (fSnapinData.FileDescription);
    result := S_OK;
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinAbout.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinAbout.GetSnapinImage(out hAppIcon: wireHICON): HResult;
var
  ico : HIcon;
begin
  try
    result := S_FALSE;
    ico := LoadIcon (HInstance, MakeIntResource (1));
    if ico <> 0 then
    begin
      hAppIcon := wireHICON (ico);
      result := S_OK
    end
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinAbout.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinAbout.GetSnapinVersion(out lpVersion: PWideChar): HResult;
begin
  try
    LoadSnapinData;
    lpVersion := StringToLPOleStr (fSnapinData.FileVersion);
    result := S_OK;
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinAbout.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
function TSnapinAbout.GetStaticFolderImage(out hSmallImage,
  hSmallImageOpen, hLargeImage: wireHBITMAP; out cMask: UINT): HResult;
var
  smBitmap : TBitmap;
  bmp : TBitmap;
begin
  try
    LoadSnapinData;
    smBitmap := Nil;
    bmp := TBitmap.Create;
    cMask := RGB (123, 234, 235);
    try
      smBitmap := TBitmap.Create;
      if Assigned (fSnapinData.ScopeItem) and Assigned (fSnapinData.ScopeSmallImages) and (fSnapinData.ScopeItem.ImageIndex > -1) and (fSnapinData.ScopeItem.ImageIndex < fSnapinData.ScopeSmallImages.Count) then
      begin
        fSnapinData.ScopeSmallImages.GetBitmap (fSnapinData.ScopeItem.ImageIndex, smBitmap);
        bmp.Assign (smBitmap);
        hSmallImage := wireHBITMAP (bmp.ReleaseHandle)
      end
      else
        hSmallImage := Nil;

      if Assigned (fSnapinData.ScopeItem) and Assigned (fSnapinData.ScopeSmallImages) and (fSnapinData.ScopeItem.OpenImageIndex > -1) and (fSnapinData.ScopeItem.OpenImageIndex < fSnapinData.ScopeSmallImages.Count) then
      begin
        fSnapinData.ScopeSmallImages.GetBitmap (fSnapinData.ScopeItem.OpenImageIndex, smBitmap);
        bmp.Assign (smBitmap);
        hSmallImageOpen := wireHBITMAP (bmp.ReleaseHandle)
      end
      else
        hSmallImageOpen := Nil;

      if Assigned (fSnapinData.ScopeItem) and Assigned (fSnapinData.ScopeLargeImages) and (fSnapinData.ScopeItem.ImageIndex > -1) and (fSnapinData.ScopeItem.ImageIndex < fSnapinData.ScopeLargeImages.Count) then
      begin
        fSnapinData.ScopeLargeImages.GetBitmap (fSnapinData.ScopeItem.ImageIndex, smBitmap);
        bmp.Assign (smBitmap);
        hLargeImage := wireHBITMAP (bmp.ReleaseHandle)
      end
      else
        hLargeImage := wireHBITMAP (LoadBitmap (HInstance, PChar (2)));

      result := S_OK;
    finally
      smBitmap.Free;
      bmp.Free;
    end
  except
    result := HandleException
  end
end;

(*--------------------------------------------------------------------------*
 | function TSnapinAbout.
 |                                                                          |
 |
 |                                                                          |
 | Parameters:                                                              |
 |
 |                                                                          |
 | The function returns an OLE success code
 *--------------------------------------------------------------------------*)
procedure TSnapinAbout.LoadSnapinData;
begin
  if not Assigned (fSnapinData) then
    fSnapinData := GetSnapinData
end;

procedure TSnapinComponentData.Update (Item : TObject);
var
  dataObject : IDataObject;
  resultItem : TResultItem;
begin
  if Item is TScopeItem then
  begin
    ComponentDataQueryDataObject (Integer (Item), 0, dataObject);
    fConsole.UpdateAllViews  (dataObject, TScopeItem (item).ResultItems.Count, 0)
  end
  else
  begin
    resultItem := item as TResultItem;
    ComponentDataQueryDataObject (Integer (resultItem), 0, dataObject);
    fConsole.UpdateAllViews  (dataObject, 1, 0)
  end

end;

end.
