unit N3ServerMonitorMainImpl;

{$WARN SYMBOL_PLATFORM OFF}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ActiveX, AxCtrls, N3ServerMonitor_TLB, StdVcl, StdCtrls, ExtCtrls, MJTrayIcon,
  ComCtrls, ActnList, Menus, ImgList, ShellAPI, IniFiles;

type
  TN3ServerMonitorMain = class(TActiveForm, IN3ServerMonitorMain)
    TrayIcon: TMJTrayIcon;
    Timer: TTimer;
    Panel: TPanel;
    ProcessList: TListView;
    Button_CloseProcess: TButton;
    Button_KillProcess: TButton;
    PopupMenu_ProcessList: TPopupMenu;
    ActionList: TActionList;
    Action_CloseProcess: TAction;
    Action_KillProcess: TAction;
    Menu_CloseProcess: TMenuItem;
    Menu_KillProcess: TMenuItem;
    PopupMenu_Tray: TPopupMenu;
    ImageList: TImageList;
    ImageListTrayProcessList: TImageList;
    ImageListProcessList: TImageList;
    procedure TimerTimer(Sender: TObject);
    procedure ActiveFormCreate(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure Action_CloseProcessExecute(Sender: TObject);
    procedure Action_KillProcessExecute(Sender: TObject);
    procedure ProcessListSelectItem(Sender: TObject; Item: TListItem;
      Selected: Boolean);
    procedure PopupMenu_TrayPopup(Sender: TObject);
    function GetProcessID(Sender: TObject): Cardinal;
    procedure ProcessListGetImageIndex(Sender: TObject; Item: TListItem);
    procedure ProcessListDblClick(Sender: TObject);
  private
    { Private declarations }
    FEvents: IN3ServerMonitorMainEvents;
    FFileIconList: THashedStringList;
    procedure ActivateEvent(Sender: TObject);
    procedure ClickEvent(Sender: TObject);
    procedure CreateEvent(Sender: TObject);
    procedure DblClickEvent(Sender: TObject);
    procedure DeactivateEvent(Sender: TObject);
    procedure DestroyEvent(Sender: TObject);
    procedure KeyPressEvent(Sender: TObject; var Key: Char);
    procedure MouseEnterEvent(Sender: TObject);
    procedure MouseLeaveEvent(Sender: TObject);
    procedure PaintEvent(Sender: TObject);
  protected
    { Protected declarations }
    procedure DefinePropertyPages(DefinePropertyPage: TDefinePropertyPage); override;
    procedure EventSinkChanged(const EventSink: IUnknown); override;
    function Get_Active: WordBool; safecall;
    function Get_AlignDisabled: WordBool; safecall;
    function Get_AlignWithMargins: WordBool; safecall;
    function Get_AutoScroll: WordBool; safecall;
    function Get_AutoSize: WordBool; safecall;
    function Get_AxBorderStyle: TxActiveFormBorderStyle; safecall;
    function Get_Caption: WideString; safecall;
    function Get_Color: OLE_COLOR; safecall;
    function Get_DockSite: WordBool; safecall;
    function Get_DoubleBuffered: WordBool; safecall;
    function Get_DropTarget: WordBool; safecall;
    function Get_Enabled: WordBool; safecall;
    function Get_ExplicitHeight: Integer; safecall;
    function Get_ExplicitLeft: Integer; safecall;
    function Get_ExplicitTop: Integer; safecall;
    function Get_ExplicitWidth: Integer; safecall;
    function Get_Font: IFontDisp; safecall;
    function Get_HelpFile: WideString; safecall;
    function Get_KeyPreview: WordBool; safecall;
    function Get_MouseInClient: WordBool; safecall;
    function Get_PixelsPerInch: Integer; safecall;
    function Get_PopupMode: TxPopupMode; safecall;
    function Get_PrintScale: TxPrintScale; safecall;
    function Get_Scaled: WordBool; safecall;
    function Get_ScreenSnap: WordBool; safecall;
    function Get_SnapBuffer: Integer; safecall;
    function Get_UseDockManager: WordBool; safecall;
    function Get_Visible: WordBool; safecall;
    function Get_VisibleDockClientCount: Integer; safecall;
    procedure _Set_Font(var Value: IFontDisp); safecall;
    procedure Set_AlignWithMargins(Value: WordBool); safecall;
    procedure Set_AutoScroll(Value: WordBool); safecall;
    procedure Set_AutoSize(Value: WordBool); safecall;
    procedure Set_AxBorderStyle(Value: TxActiveFormBorderStyle); safecall;
    procedure Set_Caption(const Value: WideString); safecall;
    procedure Set_Color(Value: OLE_COLOR); safecall;
    procedure Set_DockSite(Value: WordBool); safecall;
    procedure Set_DoubleBuffered(Value: WordBool); safecall;
    procedure Set_DropTarget(Value: WordBool); safecall;
    procedure Set_Enabled(Value: WordBool); safecall;
    procedure Set_Font(const Value: IFontDisp); safecall;
    procedure Set_HelpFile(const Value: WideString); safecall;
    procedure Set_KeyPreview(Value: WordBool); safecall;
    procedure Set_PixelsPerInch(Value: Integer); safecall;
    procedure Set_PopupMode(Value: TxPopupMode); safecall;
    procedure Set_PrintScale(Value: TxPrintScale); safecall;
    procedure Set_Scaled(Value: WordBool); safecall;
    procedure Set_ScreenSnap(Value: WordBool); safecall;
    procedure Set_SnapBuffer(Value: Integer); safecall;
    procedure Set_UseDockManager(Value: WordBool); safecall;
    procedure Set_Visible(Value: WordBool); safecall;
  public
    { Public declarations }
    procedure Initialize; override;
    destructor Destroy; override;
  end;

type
  TWindowsVersion = (wvUnknown, wvWin95, wvWin95OSR2, wvWin98, wvWin98SE, wvWinME,
                     wvWinNT31, wvWinNT35, wvWinNT351, wvWinNT4, wvWin2000, wvWinXP, wvWin2003);

  TProcType = (ptNoCheck, ptUnknown, ptServer, ptClient, ptAdmin, ptUnknownEntry, ptSuspectEntry);

  PProcListEntry = ^TProcListEntry;
  TProcListEntry = record
    PID: Cardinal;
    Name: String;
    ProcType: TProcType;
  end;
  TProcList = array of TProcListEntry;

const
  TProcTypeNames: array[TProcType] of string = ('NoCheck', 'Unknown', 'Server', 'Client', 'Admin', 'UnknownEntry', 'SuspectEntry');

var
  CashProcList: TProcList;
  CashProcListCS: TRTLCriticalSection;

  IsWin95: Boolean = False;
  IsWin95OSR2: Boolean = False;
  IsWin98: Boolean = False;
  IsWin98SE: Boolean = False;
  IsWinME: Boolean = False;
  IsWinNT: Boolean = False;
  IsWinNT3: Boolean = False;
  IsWinNT31: Boolean = False;
  IsWinNT35: Boolean = False;
  IsWinNT351: Boolean = False;
  IsWinNT4: Boolean = False;
  IsWin2K: Boolean = False;
  IsWinXP: Boolean = False;
  IsWin2003: Boolean = False;

  ProcessorCount: Cardinal = 0;
  AllocGranularity: Cardinal = 0;
  PageSize: Cardinal = 0;

resourcestring
  RsSystemProcess = 'System Process';
  RsSystemIdleProcess = 'System Idle Process';

implementation

uses
  ComObj, ComServ, ShlObj, TLHelp32, PsApi, unitVersionInfo, Unit1;

{$R *.DFM}

var
  KernelVersionHi: DWORD;

function GetWindowsVersion: TWindowsVersion;
var
  TrimmedWin32CSDVersion: string;
begin
  Result := wvUnknown;
  TrimmedWin32CSDVersion := Trim(Win32CSDVersion);
  case Win32Platform of
    VER_PLATFORM_WIN32_WINDOWS:
      case Win32MinorVersion of
        0..9:
          if (TrimmedWin32CSDVersion = 'B') or (TrimmedWin32CSDVersion = 'C') then
            Result := wvWin95OSR2
          else
            Result := wvWin95;
        10..89:
          // On Windows ME Win32MinorVersion can be 10 (indicating Windows 98
          // under certain circumstances (image name is setup.exe). Checking
          // the kernel version is one way of working around that.
          if KernelVersionHi = $0004005A then // 4.90.x.x
            Result := wvWinME
          else
          if TrimmedWin32CSDVersion = 'A' then
            Result := wvWin98SE
          else
            Result := wvWin98;
        90:
          Result := wvWinME;
      end;
    VER_PLATFORM_WIN32_NT:
      case Win32MajorVersion of
        3:
          case Win32MinorVersion of
            1:
              Result := wvWinNT31;
            5:
              Result := wvWinNT35;
            51:
              Result := wvWinNT351;
          end;
        4:
          Result := wvWinNT4;
        5:
          case Win32MinorVersion of
            0:
              Result := wvWin2000;
            1:
              Result := wvWinXP;
            else
              Result := wvWin2003;
          end;
      end;
  end;
end;

function GetIcon(FileName: string): HICON;
var
  Sfi: TSHFileInfo;
begin
  SHGetFileInfo(PChar(FileName), 0, Sfi, SizeOf(TSHFileInfo), SHGFI_EXETYPE or SHGFI_ICON or SHGFI_SMALLICON);
  Result := Sfi.hIcon;
end;

{ TN3ServerMonitorMain }

procedure TN3ServerMonitorMain.DefinePropertyPages(DefinePropertyPage: TDefinePropertyPage);
begin
  { Define property pages here.  Property pages are defined by calling
    DefinePropertyPage with the class id of the page.  For example,
      DefinePropertyPage(Class_N3ServerMonitorMainPage); }
end;

procedure TN3ServerMonitorMain.EventSinkChanged(const EventSink: IUnknown);
begin
  FEvents := EventSink as IN3ServerMonitorMainEvents;
  inherited EventSinkChanged(EventSink);
end;

procedure TN3ServerMonitorMain.FormResize(Sender: TObject);
begin
  if (Width = 0) or (Height = 0) then
    Panel.Align := alNone
  else
    Panel.Align := alClient;
end;

procedure TN3ServerMonitorMain.Initialize;
begin
  inherited Initialize;
  OnActivate := ActivateEvent;
  OnClick := ClickEvent;
  OnCreate := CreateEvent;
  OnDblClick := DblClickEvent;
  OnDeactivate := DeactivateEvent;
  OnDestroy := DestroyEvent;
  OnKeyPress := KeyPressEvent;
  OnMouseEnter := MouseEnterEvent;
  OnMouseLeave := MouseLeaveEvent;
  OnPaint := PaintEvent;
  InitializeCriticalSection(CashProcListCS);
  FFileIconList := THashedStringList.Create();
  FFileIconList.CaseSensitive := False;
end;

function TN3ServerMonitorMain.GetProcessID(Sender: TObject): Cardinal;
var
  Item: TListItem;
begin
  if (Sender is TMenuItem) and (TMenuItem(Sender).Tag > 0) then
    Result := Cardinal(TMenuItem(Sender).Tag)
  else begin
    Item := ProcessList.Selected;
    if not Assigned(Item) then
      Result := 0
    else
      Result := StrToInt(Item.Caption);
  end;
end;

function TN3ServerMonitorMain.Get_Active: WordBool;
begin
  Result := Active;
end;

function TN3ServerMonitorMain.Get_AlignDisabled: WordBool;
begin
  Result := AlignDisabled;
end;

function TN3ServerMonitorMain.Get_AlignWithMargins: WordBool;
begin
  Result := AlignWithMargins;
end;

function TN3ServerMonitorMain.Get_AutoScroll: WordBool;
begin
  Result := AutoScroll;
end;

function TN3ServerMonitorMain.Get_AutoSize: WordBool;
begin
  Result := AutoSize;
end;

function TN3ServerMonitorMain.Get_AxBorderStyle: TxActiveFormBorderStyle;
begin
  Result := Ord(AxBorderStyle);
end;

function TN3ServerMonitorMain.Get_Caption: WideString;
begin
  Result := WideString(Caption);
end;

function TN3ServerMonitorMain.Get_Color: OLE_COLOR;
begin
  Result := OLE_COLOR(Color);
end;

function TN3ServerMonitorMain.Get_DockSite: WordBool;
begin
  Result := DockSite;
end;

function TN3ServerMonitorMain.Get_DoubleBuffered: WordBool;
begin
  Result := DoubleBuffered;
end;

function TN3ServerMonitorMain.Get_DropTarget: WordBool;
begin
  Result := DropTarget;
end;

function TN3ServerMonitorMain.Get_Enabled: WordBool;
begin
  Result := Enabled;
end;

function TN3ServerMonitorMain.Get_ExplicitHeight: Integer;
begin
  Result := ExplicitHeight;
end;

function TN3ServerMonitorMain.Get_ExplicitLeft: Integer;
begin
  Result := ExplicitLeft;
end;

function TN3ServerMonitorMain.Get_ExplicitTop: Integer;
begin
  Result := ExplicitTop;
end;

function TN3ServerMonitorMain.Get_ExplicitWidth: Integer;
begin
  Result := ExplicitWidth;
end;

function TN3ServerMonitorMain.Get_Font: IFontDisp;
begin
  GetOleFont(Font, Result);
end;

function TN3ServerMonitorMain.Get_HelpFile: WideString;
begin
  Result := WideString(HelpFile);
end;

function TN3ServerMonitorMain.Get_KeyPreview: WordBool;
begin
  Result := KeyPreview;
end;

function TN3ServerMonitorMain.Get_MouseInClient: WordBool;
begin
  Result := MouseInClient;
end;

function TN3ServerMonitorMain.Get_PixelsPerInch: Integer;
begin
  Result := PixelsPerInch;
end;

function TN3ServerMonitorMain.Get_PopupMode: TxPopupMode;
begin
  Result := Ord(PopupMode);
end;

function TN3ServerMonitorMain.Get_PrintScale: TxPrintScale;
begin
  Result := Ord(PrintScale);
end;

function TN3ServerMonitorMain.Get_Scaled: WordBool;
begin
  Result := Scaled;
end;

function TN3ServerMonitorMain.Get_ScreenSnap: WordBool;
begin
  Result := ScreenSnap;
end;

function TN3ServerMonitorMain.Get_SnapBuffer: Integer;
begin
  Result := SnapBuffer;
end;

function TN3ServerMonitorMain.Get_UseDockManager: WordBool;
begin
  Result := UseDockManager;
end;

function TN3ServerMonitorMain.Get_Visible: WordBool;
begin
  Result := Visible;
end;

function TN3ServerMonitorMain.Get_VisibleDockClientCount: Integer;
begin
  Result := VisibleDockClientCount;
end;

procedure TN3ServerMonitorMain._Set_Font(var Value: IFontDisp);
begin
  SetOleFont(Font, Value);
end;

procedure TN3ServerMonitorMain.Action_CloseProcessExecute(Sender: TObject);

  function EnumWindowsProc(Wnd: HWND; ProcessID: DWORD): Boolean; stdcall;
  var
    PID: DWORD;
  begin
    GetWindowThreadProcessId(Wnd, @PID);
    if ProcessID = PID then
      PostMessage(Wnd, WM_CLOSE, 0, 0);
    Result := True;
  end;

var
  ProcessID: Cardinal;
  HProc: THandle;
begin
  ProcessID := GetProcessID(Sender);
  if ProcessID = 0 then
    Exit;
  HProc := OpenProcess(SYNCHRONIZE or PROCESS_TERMINATE, False, ProcessID);
  if HProc <> 0 then
  try
    EnumWindows(@EnumWindowsProc, LPARAM(ProcessID));
    if WaitForSingleObject(HProc, 5000) <> WAIT_OBJECT_0 then
      MessageBox(0, '  !', nil, MB_ICONERROR);
  finally
    CloseHandle(HProc);
  end;
end;

procedure TN3ServerMonitorMain.Action_KillProcessExecute(Sender: TObject);
var
  ProcessID: Cardinal;
  HProc: THandle;
begin
  ProcessID := GetProcessID(Sender);
  if ProcessID = 0 then
    Exit;
  HProc := OpenProcess(PROCESS_TERMINATE, False, ProcessID);
  if HProc <> 0 then
  try
    if not TerminateProcess(HProc, 0) then
      MessageBox(0, '  !', nil, MB_ICONERROR);
  finally
    CloseHandle(HProc);
  end;
end;

procedure TN3ServerMonitorMain.ActivateEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnActivate;
end;

procedure TN3ServerMonitorMain.ActiveFormCreate(Sender: TObject);
begin
  OnResize := FormResize;
end;

procedure TN3ServerMonitorMain.ClickEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnClick;
end;

procedure TN3ServerMonitorMain.CreateEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnCreate;
end;

procedure TN3ServerMonitorMain.DblClickEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnDblClick;
end;

procedure TN3ServerMonitorMain.DeactivateEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnDeactivate;
end;

destructor TN3ServerMonitorMain.Destroy;
begin
  FFileIconList.Free();
  DeleteCriticalSection(CashProcListCS);
  inherited;
end;

procedure TN3ServerMonitorMain.DestroyEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnDestroy;
end;

procedure TN3ServerMonitorMain.KeyPressEvent(Sender: TObject; var Key: Char);
var
  TempKey: Smallint;
begin
  TempKey := Smallint(Key);
  if FEvents <> nil then FEvents.OnKeyPress(TempKey);
  Key := Char(TempKey);
end;

procedure TN3ServerMonitorMain.MouseEnterEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnMouseEnter;
end;

procedure TN3ServerMonitorMain.MouseLeaveEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnMouseLeave;
end;

procedure TN3ServerMonitorMain.PaintEvent(Sender: TObject);
begin
  if FEvents <> nil then FEvents.OnPaint;
end;

procedure TN3ServerMonitorMain.PopupMenu_TrayPopup(Sender: TObject);
var
  I: Integer;
  Item: TMenuItem;
  SubItem: TMenuItem;
  Icon: TIcon;

begin
  EnterCriticalSection(CashProcListCS);
  try
    PopupMenu_Tray.Items.Clear();
    ImageListTrayProcessList.Assign(ImageList);
    for I := 0 to Length(CashProcList) - 1 do
    begin
      if CashProcList[I].ProcType = ptUnknown then
        Continue;
      Item := TMenuItem.Create(PopupMenu_Tray.Items);
      Item.Caption := ExtractFileName(CashProcList[I].Name) + ' (' + IntToStr(CashProcList[I].PID) + ')';
      Icon := TIcon.Create();
      try
        Icon.Handle := GetIcon(CashProcList[I].Name);
        Item.ImageIndex := ImageListTrayProcessList.AddIcon(Icon);
      except
      end;
      Icon.Free();
      SubItem := TMenuItem.Create(Item);
      SubItem.Tag := CashProcList[I].PID;
      SubItem.Action := Action_CloseProcess;
      SubItem.Action := nil;
      SubItem.Enabled := True;
      Item.Add(SubItem);
      SubItem := TMenuItem.Create(Item);
      SubItem.Tag := CashProcList[I].PID;
      SubItem.Action := Action_KillProcess;
      SubItem.Action := nil;
      SubItem.Enabled := True;
      Item.Add(SubItem);
      PopupMenu_Tray.Items.Add(Item);
    end;
  finally
    LeaveCriticalSection(CashProcListCS);
  end;
end;

procedure TN3ServerMonitorMain.ProcessListDblClick(Sender: TObject);
begin
  Action_KillProcess.Execute();
end;

procedure TN3ServerMonitorMain.ProcessListGetImageIndex(Sender: TObject;
  Item: TListItem);
var
  I: Integer;
  Icon: TIcon;
  FN: string;
begin
  if Item.ImageIndex < 0 then
  begin
    FN := PProcListEntry(Item.Data)^.Name;
    I := FFileIconList.IndexOf(FN);
    if I < 0 then
    begin
      Icon := TIcon.Create();
      try
        Icon.Handle := GetIcon(FN);
        I := ImageListProcessList.AddIcon(Icon);
      except
        I := -1;
      end;
      Icon.Free();
      I := FFileIconList.AddObject(FN, TObject(I));
    end;
    Item.ImageIndex := Integer(FFileIconList.Objects[I]);
  end;
end;

procedure TN3ServerMonitorMain.ProcessListSelectItem(Sender: TObject;
  Item: TListItem; Selected: Boolean);
var
  B: Boolean;
begin
  B := Selected and (Item <> nil);
  Action_CloseProcess.Enabled := B;
  Action_KillProcess.Enabled := B;
end;

procedure TN3ServerMonitorMain.Set_AlignWithMargins(Value: WordBool);
begin
  AlignWithMargins := Value;
end;

procedure TN3ServerMonitorMain.Set_AutoScroll(Value: WordBool);
begin
  AutoScroll := Value;
end;

procedure TN3ServerMonitorMain.Set_AutoSize(Value: WordBool);
begin
  AutoSize := Value;
end;

procedure TN3ServerMonitorMain.Set_AxBorderStyle(
  Value: TxActiveFormBorderStyle);
begin
  AxBorderStyle := TActiveFormBorderStyle(Value);
end;

procedure TN3ServerMonitorMain.Set_Caption(const Value: WideString);
begin
  Caption := TCaption(Value);
end;

procedure TN3ServerMonitorMain.Set_Color(Value: OLE_COLOR);
begin
  Color := TColor(Value);
end;

procedure TN3ServerMonitorMain.Set_DockSite(Value: WordBool);
begin
  DockSite := Value;
end;

procedure TN3ServerMonitorMain.Set_DoubleBuffered(Value: WordBool);
begin
  DoubleBuffered := Value;
end;

procedure TN3ServerMonitorMain.Set_DropTarget(Value: WordBool);
begin
  DropTarget := Value;
end;

procedure TN3ServerMonitorMain.Set_Enabled(Value: WordBool);
begin
  Enabled := Value;
end;

procedure TN3ServerMonitorMain.Set_Font(const Value: IFontDisp);
begin
  SetOleFont(Font, Value);
end;

procedure TN3ServerMonitorMain.Set_HelpFile(const Value: WideString);
begin
  HelpFile := string(Value);
end;

procedure TN3ServerMonitorMain.Set_KeyPreview(Value: WordBool);
begin
  KeyPreview := Value;
end;

procedure TN3ServerMonitorMain.Set_PixelsPerInch(Value: Integer);
begin
  PixelsPerInch := Value;
end;

procedure TN3ServerMonitorMain.Set_PopupMode(Value: TxPopupMode);
begin
  PopupMode := TPopupMode(Value);
end;

procedure TN3ServerMonitorMain.Set_PrintScale(Value: TxPrintScale);
begin
  PrintScale := TPrintScale(Value);
end;

procedure TN3ServerMonitorMain.Set_Scaled(Value: WordBool);
begin
  Scaled := Value;
end;

procedure TN3ServerMonitorMain.Set_ScreenSnap(Value: WordBool);
begin
  ScreenSnap := Value;
end;

procedure TN3ServerMonitorMain.Set_SnapBuffer(Value: Integer);
begin
  SnapBuffer := Value;
end;

procedure TN3ServerMonitorMain.Set_UseDockManager(Value: WordBool);
begin
  UseDockManager := Value;
end;

procedure TN3ServerMonitorMain.Set_Visible(Value: WordBool);
begin
  Visible := Value;
end;

procedure UpdateProcessInfo(List: TListView);

  procedure UpdateProcess(Info: PProcListEntry; Items: TStrings);

    procedure CleanItems;
    begin
      Items[1] := '';
    end;

  var
    HProc: THandle;
    PMI: TProcessMemoryCounters;
    Str: string;
  begin
    HProc := OpenProcess(PROCESS_QUERY_INFORMATION, False, Info^.PID);
    if HProc = 0 then
      CleanItems()
    else begin
      try
        if not GetProcessMemoryInfo(HProc, @PMI, SizeOf(PMI)) then
          CleanItems()
        else begin
          Str := IntToStr(PMI.WorkingSetSize div (1024 * 1024) + 1) + '  (' +
          IntToStr(PMI.PeakWorkingSetSize div (1024 * 1024) + 1)+ ' ) / ' +
          IntToStr(PMI.PagefileUsage div (1024 * 1024) + 1) + '  (' +
          IntToStr(PMI.PeakPagefileUsage div (1024 * 1024) + 1) + ' )';
          if Items[1] <> Str then
            Items[1] := Str;
        end;
      finally
        CloseHandle(HProc);
      end;
    end;
  end;

var
  I: Integer;
begin
  for I := 0 to Length(CashProcList) - 1 do
  try
    UpdateProcess(PProcListEntry(List.Items[I].Data), List.Items[I].SubItems);
  except
  end;
end;

function RunningProcessesList(List: TListView): Boolean;

var
  ProcList: TProcList;
  li: Integer;
  Item: TListItem;

  function ProcessFileName(PID: DWORD): string;
  var
    Handle: THandle;
  begin
    Result := '';
    Handle := OpenProcess(PROCESS_QUERY_INFORMATION or PROCESS_VM_READ, False, PID);
    if Handle <> 0 then
    try
      SetLength(Result, MAX_PATH);
      if GetModuleFileNameEx(Handle, 0, PChar(Result), MAX_PATH) > 0 then
        SetLength(Result, StrLen(PChar(Result)))
      else
        Result := '';
    finally
      CloseHandle(Handle);
    end;
  end;

  function BuildListTH: Boolean;
  var
    SnapProcHandle: THandle;
    ProcEntry: TProcessEntry32;
    NextProc: Boolean;
    FileName: string;
  begin
    SnapProcHandle := CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
    Result := (SnapProcHandle <> INVALID_HANDLE_VALUE);
    if Result then
    try
      ProcEntry.dwSize := SizeOf(ProcEntry);
      NextProc := Process32First(SnapProcHandle, ProcEntry);
      while NextProc do
      begin
        if ProcEntry.th32ProcessID = 0 then
        begin
          // PID 0 is always the "System Idle Process" but this name cannot be
          // retrieved from the system and has to be fabricated.
          FileName := RsSystemIdleProcess;
        end
        else
        begin
          FileName := ProcessFileName(ProcEntry.th32ProcessID);
          if FileName = '' then
            FileName := ProcEntry.szExeFile;
        end;
        li := Length(ProcList);
        SetLength(ProcList, li + 1);
        ProcList[li].PID := ProcEntry.th32ProcessID;
        ProcList[li].Name := FileName;
        NextProc := Process32Next(SnapProcHandle, ProcEntry);
      end;
    finally
      CloseHandle(SnapProcHandle);
    end;
  end;

  function BuildListPS: Boolean;
  var
    PIDs: array [0..1024] of DWORD;
    Needed: DWORD;
    I: Integer;
    FileName: string;
  begin
    Result := EnumProcesses(@PIDs, SizeOf(PIDs), Needed);
    if Result then
    begin
      for I := 0 to (Needed div SizeOf(DWORD)) - 1 do
      begin
        case PIDs[I] of
          0:
            // PID 0 is always the "System Idle Process" but this name cannot be
            // retrieved from the system and has to be fabricated.
            FileName := RsSystemIdleProcess;
          2:
            // On NT 4 PID 2 is the "System Process" but this name cannot be
            // retrieved from the system and has to be fabricated.
            if IsWinNT4 then
              FileName := RsSystemProcess
            else
              FileName := ProcessFileName(PIDs[I]);
          8:
            // On Win2K PID 8 is the "System Process" but this name cannot be
            // retrieved from the system and has to be fabricated.
            if IsWin2k or IsWinXP then
              FileName := RsSystemProcess
            else
              FileName := ProcessFileName(PIDs[I]);
        else
          FileName := ProcessFileName(PIDs[I]);
        end;
        if FileName <> '' then
        begin
          li := Length(ProcList);
          SetLength(ProcList, li + 1);
          ProcList[li].PID := PIDs[I];
          ProcList[li].Name := FileName;
        end;
      end;
    end;
  end;

  procedure ProcListSort();

    function SCompare(I, P: Integer): Integer;
    var
      V1, V2: Cardinal;
    begin
      V1 := ProcList[I].PID;
      V2 := ProcList[P].PID;
      if V1 = V2 then
        Result := 0
      else if V1 < V2 then
        Result := -1
      else
        Result := 1;
    end;

    procedure ExchangeItems(L, R: Integer);
    var
      T: TProcListEntry;
    begin
      T := ProcList[L];
      ProcList[L] := ProcList[R];
      ProcList[R] := T;
    end;

    procedure QuickSort(L, R: Integer);
    var
      I, J, P: Integer;
    begin
      repeat
        I := L;
        J := R;
        P := (L + R) shr 1;
        repeat
          while SCompare(I, P) < 0 do Inc(I);
          while SCompare(J, P) > 0 do Dec(J);
          if I <= J then
          begin
            ExchangeItems(I, J);
            if P = I then
              P := J
            else if P = J then
              P := I;
            Inc(I);
            Dec(J);
          end;
        until I > J;
        if L < J then QuickSort(L, J);
        L := I;
      until I >= R;
    end;

  begin
    QuickSort(0, Length(ProcList) - 1);
  end;

  function ProcListSynk: Boolean;

    function GetProcType(FileName: string): TProcType;
    var
      Handle: THandle;
      Size: DWORD;
      V: TVersionInfo;
      Buffer: string;

      function StrChk(SubStr, Str: string): Boolean;
      begin
        Result := Pos(AnsiUpperCase(SubStr), AnsiUpperCase(Str)) > 0;
      end;

    begin
      Size := GetFileVersionInfoSize(PChar(FileName), Handle);
      try
        SetLength(Buffer, Size);
        if (Size <> 0) and not GetFileVersionInfo(PChar(FileName), Handle, Size, PChar(Buffer)) then
          Buffer := '';
      finally
        CloseHandle(Handle);
      end;
      try
        if Buffer = '' then
          V := nil
        else
          V := TVersionInfo.Create(PChar(Buffer));
        try
{
'Comments' 'CompanyName' 'FileDescription' 'FileVersion' 'InternalName' 'LegalCopyright'
'LegalTradeMarks' 'OriginalFilename' 'ProductName' 'ProductVersion' 'SpecialBuild' 'PrivateBuild'
}
          FileName := ExtractFileName(FileName);
          if Assigned(V) and StrChk('N3Server', V.KeyValue['OriginalFilename']) then
            Result := ptServer
          else if Assigned(V) and StrChk('N3Client', V.KeyValue['OriginalFilename']) then
            Result := ptClient
          else if Assigned(V) and StrChk('N3CAdmin', V.KeyValue['OriginalFilename']) then
            Result := ptAdmin
          else if Assigned(V) and StrChk('N3Admin', V.KeyValue['OriginalFilename']) then
            Result := ptAdmin
          else if Assigned(V) and StrChk('N3Server', V.KeyValue['InternalName']) then
            Result := ptServer
          else if Assigned(V) and StrChk('N3Client', V.KeyValue['InternalName']) then
            Result := ptClient
          else if Assigned(V) and StrChk('N3CAdmin', V.KeyValue['InternalName']) then
            Result := ptAdmin
          else if Assigned(V) and StrChk('N3Admin', V.KeyValue['InternalName']) then
            Result := ptAdmin
          else if StrChk('N3Server', FileName) then
            Result := ptServer
          else if StrChk('N3Client', FileName) then
            Result := ptClient
          else if StrChk('N3CAdmin', FileName) then
            Result := ptAdmin
          else if StrChk('N3Admin', FileName) then
            Result := ptAdmin
          else if Assigned(V) and StrChk('Entry', V.KeyValue['CompanyName']) then
            Result := ptUnknownEntry
          else if StrChk('N3', FileName) then
            Result := ptSuspectEntry
          else
            Result := ptUnknown;
        finally
          V.Free();
        end;
      except
        Result := ptUnknown;
      end;
    end;

  var
    I: Integer;
    PID: Cardinal;
    CurrentP: Integer;
    MaxP: Integer;
    IsUpdate: Boolean;
  begin
    IsUpdate := False;
    CurrentP := 0;
    MaxP := Length(CashProcList) - 1;
    for I := 0 to Length(ProcList) - 1 do
    begin
      PID := ProcList[I].PID;
      while (CurrentP <= MaxP) and (PID > CashProcList[CurrentP].PID) do
      begin
        if CashProcList[CurrentP].ProcType <> ptUnknown then
          IsUpdate := True;
        Inc(CurrentP);
      end;
      if (CurrentP <= MaxP) and (PID = CashProcList[CurrentP].PID) then
      begin
        ProcList[I].ProcType := CashProcList[CurrentP].ProcType;
        Inc(CurrentP);
      end
      else begin
        ProcList[I].ProcType := GetProcType(ProcList[I].Name);
        if ProcList[I].ProcType <> ptUnknown then
          IsUpdate := True;
      end;
    end;
    CashProcList := ProcList;
    Result := IsUpdate;
  end;

  procedure ProcListUpdate;
  var
    I: Integer;
  begin
    List.Items.BeginUpdate();
    List.Items.Clear();
    for I := 0 to Length(CashProcList) - 1 do
    begin
      if CashProcList[I].ProcType = ptUnknown then
        Continue;
      Item := List.Items.Add();
      Item.Caption := IntToStr(CashProcList[I].PID);
      Item.SubItems.Add(ExtractFileName(CashProcList[I].Name) + ' (' + TProcTypeNames[CashProcList[I].ProcType] + ')');
      Item.SubItems.Add('');
      Item.SubItems.Add(ExtractFilePath(CashProcList[I].Name));
      Item.ImageIndex := -1;
      Item.Data := @CashProcList[I];
    end;
    List.Items.EndUpdate();
  end;

begin
  SetLength(ProcList, 0);
  if GetWindowsVersion in [wvWinNT31, wvWinNT35, wvWinNT351, wvWinNT4] then
    Result := BuildListPS
  else
    Result := BuildListTH;
  ProcListSort();
  if ProcListSynk() then
    ProcListUpdate();
end;

procedure TN3ServerMonitorMain.TimerTimer(Sender: TObject);
begin
  EnterCriticalSection(CashProcListCS);
  try
    RunningProcessesList(ProcessList);
    UpdateProcessInfo(ProcessList);
  finally
    LeaveCriticalSection(CashProcListCS);
  end;
end;

function VersionFixedFileInfo(const FileName: string; var FixedInfo: TVSFixedFileInfo): Boolean;
var
  Size, FixInfoLen: DWORD;
  Handle: THandle;
  Buffer: string;
  FixInfoBuf: PVSFixedFileInfo;
begin
  Result := False;
  Size := GetFileVersionInfoSize(PChar(FileName), Handle);
  if Size > 0 then
  begin
    SetLength(Buffer, Size);
    if GetFileVersionInfo(PChar(FileName), Handle, Size, Pointer(Buffer)) and
      VerQueryValue(Pointer(Buffer), '\', Pointer(FixInfoBuf), FixInfoLen) and
      (FixInfoLen = SizeOf(TVSFixedFileInfo)) then
    begin
      Result := True;
      FixedInfo := FixInfoBuf^;
    end;
  end;
end;

procedure InitSysInfo;
var
  SystemInfo: TSystemInfo;
  Kernel32FileName: string;
  VerFixedFileInfo: TVSFixedFileInfo;
begin
  FillChar(SystemInfo, SizeOf(SystemInfo), #0);
  GetSystemInfo(SystemInfo);
  ProcessorCount := SystemInfo.dwNumberOfProcessors;
  AllocGranularity := SystemInfo.dwAllocationGranularity;
  PageSize := SystemInfo.dwPageSize;

  { Windows version information }

  IsWinNT := Win32Platform = VER_PLATFORM_WIN32_NT;

  Kernel32FileName := GetModuleName(GetModuleHandle(kernel32));
  if (not IsWinNT) and VersionFixedFileInfo(Kernel32FileName, VerFixedFileInfo) then
    KernelVersionHi := VerFixedFileInfo.dwProductVersionMS
  else
    KernelVersionHi := 0;

  case GetWindowsVersion of
    wvUnknown: ;
    wvWin95:
      IsWin95 := True;
    wvWin95OSR2:
      IsWin95OSR2 := True;
    wvWin98:
      IsWin98 := True;
    wvWin98SE:
      IsWin98SE := True;
    wvWinME:
      IsWinME := True;
    wvWinNT31:
      begin
        IsWinNT3 := True;
        IsWinNT31 := True;
      end;
    wvWinNT35:
      begin
        IsWinNT3 := True;
        IsWinNT35 := True;
      end;
    wvWinNT351:
      begin
        IsWinNT3 := True;
        IsWinNT351 := True;
      end;
    wvWinNT4:
      IsWinNT4 := True;
    wvWin2000:
      IsWin2K := True;
    wvWinXP:
      IsWinXP := True;
    wvWin2003:
      IsWin2003 := True;
  end;
end;

initialization
  TActiveFormFactory.Create(
    ComServer,
    TActiveFormControl,
    TN3ServerMonitorMain,
    Class_N3ServerMonitorMain,
    1,
    '',
    OLEMISC_SIMPLEFRAME or OLEMISC_ACTSLIKELABEL,
    tmApartment);
  InitSysInfo();

end.
