{
    This file is a part of the graphics library GraphiX
    Copyright (C) 2001 Michael Knapp

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}

{============================================================================}
{ S3 Vision, Trio, ViRGE                                                                   }
{============================================================================}
{$I gxglobal.cfg}
UNIT gxhw_s3;

INTERFACE

USES gxbase,gxpci,gx3dtype,gxsup,gxmem;
{$I gxlocal.cfg}

PROCEDURE sethwaprocs_s3vision(col:longint);
PROCEDURE sethwaprocs_s3virge(col:longint);
PROCEDURE setupdevice_s3(baseaddr:dword;var pcidev:TPCIdevice);
PROCEDURE adjustaddresses_s3(old_ds_base,new_ds_base:dword);
PROCEDURE setreslist_s3vision(mem,chipid:dword;var modelist:PModeEntry);
PROCEDURE setreslist_s3trio(mem,chipid:dword;var modelist:PModeEntry);
PROCEDURE setreslist_s3virge(mem,chipid:dword;var modelist:PModeEntry);
FUNCTION detectchip_s3(var chipid:dword;var name:string):longint;
FUNCTION detectmem_s3:longint;
PROCEDURE enableregs_s3;
PROCEDURE enableregs_s3virge;
PROCEDURE disableregs_s3;
PROCEDURE disableregs_s3virge;
PROCEDURE setcolormode_s3virge(col:longint);
PROCEDURE scanline_s3(linewidth:word);
PROCEDURE displaystart_s3(offs:dword);
PROCEDURE bankswitch_s3a;
PROCEDURE bankswitch_s3b;
PROCEDURE enableLFB_s3(physaddr:dword);
PROCEDURE enableLFB_s3_pci(physaddr:dword);
PROCEDURE disableLFB_s3;
PROCEDURE disableLFB_s3_pci;
PROCEDURE enableMMIO_s3virge(physaddr:dword);
PROCEDURE disableMMIO_s3virge;
PROCEDURE enableHWA_s3vision(col,init,bpl,bpp,pix:longint);
PROCEDURE enableHWA_s3virge(col,init,bpl,bpp,pix:longint);
PROCEDURE disableHWA_s3vision;
PROCEDURE disableHWA_s3virge;

IMPLEMENTATION

USES graphix,gxtype,gxcrtext;

VAR mmiomap0:tmmiomap;
    reg_mmio:dword;
{    patbuf:array[0..63] of byte; }


PROCEDURE setupdevice_s3(baseaddr:dword;var pcidev:TPCIdevice);
BEGIN
  CASE pcidev.device OF
  $8810..$8815,$8880..$88F0,$8901..$890F:
    BEGIN
      mapresource(mmiomap0,pcidev.base0 AND $FF800000,getpcibasesize(baseaddr,0));
      lfbbase:=mmiomap0.offs;
    END;
  $5631,$883D,$8A01,$8A10,$8A12,$8C01:
    BEGIN
      mapresource(mmiomap0,pcidev.base0 AND $FC000000,getpcibasesize(baseaddr,0));
      lfbbase:=mmiomap0.offs+$00000000;
      reg_mmio:=mmiomap0.offs+$01000000;
    END;
  END;
END;

PROCEDURE adjustaddresses_s3(old_ds_base,new_ds_base:dword);
BEGIN
  adjustaddress(lfbbase,old_ds_base,new_ds_base);
  adjustaddress(reg_mmio,old_ds_base,new_ds_base);
END;

{----------------------------------------------------------------------------}

FUNCTION detectchip_s3(var chipid:dword;var name:string):longint;
VAR chip1,chip2:word;
BEGIN
  ASM
    MOV DX,03D4h  {disable extensions}
    MOV AX,0038h
    OUT DX,AX

    MOV AL,35h
    OUT DX,AL
    INC DX
    IN AL,DX
    MOV CL,AL
    MOV AL,0Fh
    OUT DX,AL
    IN AL,DX
    XCHG AL,CL
    OUT DX,AL
    DEC DX

    CMP CL,0Fh
    JE @s3_not_detected

    MOV AX,4838h
    OUT DX,AX

    MOV AL,35h
    OUT DX,AL
    INC DX
    IN AL,DX
    MOV CL,AL
    MOV AL,0Fh
    OUT DX,AL
    IN AL,DX
    XCHG AL,CL
    OUT DX,AL
    DEC DX

    CMP CL,0Fh
    JNE @s3_not_detected

    MOV AL,30h
    OUT DX,AL
    INC DX
    IN AL,DX
    DEC DX
    MOV AH,AL
    XOR AL,AL

    CMP AH,0E0h
    JNE @s3_detected
    CMP AH,0E1h
    JNE @s3_detected

    MOV AL,2Eh
    OUT DX,AL
    INC DX
    IN AL,DX
    DEC DX
    MOV BL,AL

    MOV AL,2Fh
    OUT DX,AL
    INC DX
    IN AL,DX
    XCHG AL,BL
    JMP @s3_detected
  @s3_not_detected:
    MOV AX,0
    XOR DX,DX
    JMP @ende
  @s3_detected:
    MOV DH,BL
    XCHG AL,AH
    XCHG AH,DH
    XCHG AL,AH
  @ende:
    MOV chip1,AX
    MOV chip2,DX
  END; { AL=2Eh, AH=30h, DH=2Fh}
  CASE chip1 OF
    $0000:;
    $8100..$81FF:name:='S3 86c911';
    $8200..$82FF:name:='S3 86c911A/924';
    $9000..$90FF:name:='S3 86c928';
    $9100..$91FF:name:='S3 86c928C';
    $9400..$94FF:name:='S3 86c928D';
    $9500..$95FF:name:='S3 86c928E';
    $A000..$A0FF:name:='S3 86c801/805AB';
    $A200..$A4FF:name:='S3 86c801/805C';
    $A500..$A5FF:name:='S3 86c801/805D';
    $A600..$A6FF:name:='S3 86c801/805P';
    $A700..$A7FF:name:='S3 86c801/805D';
    $A800..$A8FF:name:='S3 86c801/805I';
    $B000..$B0FF:name:='S3 86c928 PCI';
    $C000..$C0FF:name:='S3 86c864 VLB - Vision864';
    $C100..$C1FF:name:='S3 86c864 PCI - Vision864';
    $D000..$D0FF:name:='S3 86c964 VLB - Vision964';
    $D100..$D1FF:name:='S3 86c964 PCI - Vision964';
    $E010,$E110:name:='S3 86c732 - Trio32';
    $E011,$E111:CASE (chip2 AND $FF00) OF
                  $5000:name:='S3 86c765 - Trio64V+';
                  ELSE name:='S3 86c764 - Trio64';
                END;
    $E031,$E131:name:='S3 86c325 - ViRGE';
    $E03D,$E13D:name:='S3 86c988 - ViRGE/VX';
    $E080,$E180:name:='S3 86c866';
    $E090,$E190:name:='S3 86c868';
    $E0B0,$E1B0:name:='S3 86c375 - ViRGE/DX';
    $E0F0,$E1F0:name:='S3 86c968';
    ELSE name:='S3 - unknown - ID '+hexword(chip1)+'-'+hexword(chip2);
  END;
  chipid:=chip1;
  detectchip_s3:=chip1;
END;

FUNCTION detectmem_s3:longint;
VAR mem:word;
BEGIN
  ASM
    MOV DX,03D4h
    MOV AL,36h
    OUT DX,AL
    INC DX
    IN AL,DX
    AND AX,00E0h
    MOV mem,AX
  END;
  CASE mem OF
    $0000:detectmem_s3:=4096;
    $0020:detectmem_s3:=512;
    $0040:detectmem_s3:=3072;
    $0060:detectmem_s3:=8192;
    $0080:detectmem_s3:=2048;
    $00A0:detectmem_s3:=6144;
    $00C0:detectmem_s3:=1024;
    $00E0:detectmem_s3:=512;
  END;
END;


{----------------------------------------------------------------------------}

PROCEDURE enableregs_s3;assembler;
ASM
{enable extensions}
  MOV DX,03D4h
  MOV AX,4838h
  OUT DX,AX
  MOV AX,0A539h
  OUT DX,AX
{clear old logical screen width bit}
  MOV AL,43h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0FBh
  OUT DX,AL
{little speed increase}
{  MOV AX,0FF68h
   OUT DX,AX }
{enable enhanced mode}
{  MOV DX,03D4h
   MOV AX,0931h
   OUT DX,AX }
END;

PROCEDURE enableregs_s3virge;assembler;
ASM
{enable extensions}
  MOV DX,03D4h
  MOV AX,4838h
  OUT DX,AX
  MOV AX,0A539h
  OUT DX,AX
{clear old logical screen width bit}
  MOV AL,43h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0FBh
  OUT DX,AL
{wait until retrace starts}
  MOV DX,03DAh
@rsl2:
  IN AL,DX
  TEST AL,08h
  JZ @rsl2
{enable streams proccessor}
{  MOV DX,03D4h
  MOV AL,67h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,03h
  OR AL,0Ch
  OUT DX,AL }
END;

PROCEDURE disableregs_s3;assembler;
ASM
  MOV DX,03D4h
  MOV AX,0038h
  OUT DX,AX
  MOV AX,5A39h
  OUT DX,AX
END;

PROCEDURE disableregs_s3virge;assembler;
ASM
{wait until retrace starts}
  MOV DX,03DAh
@rsl2:
  IN AL,DX
  TEST AL,08h
  JZ @rsl2
{disable streams processor}
  MOV DX,03D4h
  MOV AL,67h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0F3h
  OUT DX,AL
{disable regs}
  MOV DX,03D4h
  MOV AX,0038h
  OUT DX,AX
  MOV AX,5A39h
  OUT DX,AX
END;

PROCEDURE setcolormode_s3virge(col:longint);
VAR yd:longint;
BEGIN
  ASM
  {disable streams processor}
    MOV DX,03D4h
    MOV AL,67h
    OUT DX,AL
    INC DX
    IN AL,DX
    AND AL,0F3h
    OUT DX,AL
  END;
  CASE col OF
  ig_col24:
    BEGIN
      ASM
      {enable streams processor}
        MOV DX,03D4h
        MOV AL,67h
        OUT DX,AL
        INC DX
        IN AL,DX
        AND AL,03h
        OR AL,0Ch
        OUT DX,AL
      END;
      yd:=maxY+1;
      IF (yd<=384) THEN yd:=yd*2;
      dword(pointer(reg_mmio+$8180)^):=$06000000;
      dword(pointer(reg_mmio+$8184)^):=$00000000;
      dword(pointer(reg_mmio+$8190)^):=$03000000;
      dword(pointer(reg_mmio+$8194)^):=$00000000;
      dword(pointer(reg_mmio+$8198)^):=$00000000;
      dword(pointer(reg_mmio+$81A0)^):=$01000000;
      dword(pointer(reg_mmio+$81C0)^):=$00000000;
      dword(pointer(reg_mmio+$81C4)^):=$00000000;
      dword(pointer(reg_mmio+$81C8)^):=bytperline;
      dword(pointer(reg_mmio+$81CC)^):=$00000000;
      dword(pointer(reg_mmio+$81D0)^):=$00000000;
      dword(pointer(reg_mmio+$81D4)^):=$00000000;
      dword(pointer(reg_mmio+$81D8)^):=$00000001;
      dword(pointer(reg_mmio+$81DC)^):=$40000000;
      dword(pointer(reg_mmio+$81E0)^):=$00000000;
      dword(pointer(reg_mmio+$81E4)^):=$00000000;
      dword(pointer(reg_mmio+$81E8)^):=$00000000;
      dword(pointer(reg_mmio+$81F0)^):=$00010001;
      dword(pointer(reg_mmio+$81F4)^):=(maxX SHL 16)+yd;
      dword(pointer(reg_mmio+$81F8)^):=$07FF07FF;
      dword(pointer(reg_mmio+$81FC)^):=$00010001;
    END;
  END;
END;

PROCEDURE scanline_s3(linewidth:word);assembler;
ASM
{bit 0-7}
  MOV DX,03D4h
  MOV AH,BYTE PTR linewidth
  MOV AL,13h
  OUT DX,AX
{bit 8-9}
  MOV AX,linewidth
  SHL AH,4
  MOV AL,51h
  OUT DX,AX
END;

PROCEDURE displaystart_s3(offs:dword);assembler;
ASM
  MOV EBX,offs
{bit 0-7}
  SHR EBX,2
  MOV DX,03D4h
  MOV AH,BL
  MOV AL,0Dh
  OUT DX,AX
{bit 8-15}
  SHR EBX,8
  MOV DX,03D4h
  MOV AH,BL
  MOV AL,0Ch
  OUT DX,AX
{bit 16-19}
  SHR EBX,8
  AND BL,0Fh
  MOV AL,69h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0F0h
  OR AL,BL
  OUT DX,AL
END;

PROCEDURE bankswitch_s3a;assembler;
ASM
  MOV AH,DL
  MOV BL,DL
  AND AH,0Fh
  MOV AL,35h
  MOV DX,03D4h
  OUT DX,AX

  SHR BL,2
  AND BL,0Ch

  MOV DX,03D4h
  MOV AL,51h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0F3h
  OR AL,BL
  OUT DX,AL
END;

PROCEDURE bankswitch_s3b;assembler;
ASM
  MOV AH,DL
  MOV AL,6Ah
  MOV DX,03D4h
  OUT DX,AX
END;

PROCEDURE enableLFB_s3(physaddr:dword);assembler;
ASM
  MOV DX,03D4h {disable LFB}
  MOV AL,58h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0ECh
  OUT DX,AL
  DEC DX

  MOV BX,WORD PTR [physaddr+2]
  MOV AL,59h
  MOV AH,BH
  OUT DX,AX
  MOV AL,5Ah
  MOV AH,BL
  OUT DX,AX

  MOV DX,03D4h {enable LFB}
  MOV AL,58h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0ECh
  OR AL,13h
  OUT DX,AL
END;

PROCEDURE enableLFB_s3_pci(physaddr:dword);assembler;
ASM
  MOV DX,03D4h {enable LFB}
  MOV AL,58h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0ECh
  OR AL,13h
  OUT DX,AL
END;

PROCEDURE disableLFB_s3;assembler;
ASM
  MOV DX,03D4h {disable LFB}
  MOV AL,58h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0ECh
  OUT DX,AL
  DEC DX

  MOV AX,0059h
  OUT DX,AX
  MOV AX,0A5Ah
  OUT DX,AX
END;

PROCEDURE disableLFB_s3_pci;assembler;
ASM
  MOV DX,03D4h {disable LFB}
  MOV AL,58h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0ECh
  OUT DX,AL
END;

PROCEDURE enableMMIO_s3virge(physaddr:dword);assembler;
ASM
  MOV DX,03D4h
  MOV AL,53h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0C1h
  OR AL,08h
  OUT DX,AL
END;

PROCEDURE disableMMIO_s3virge;assembler;
ASM
  MOV DX,03D4h
  MOV AL,53h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0C1h
  OUT DX,AL
END;

{------------------------------- Hardware-Acceleration ----------------------}

VAR crtc66hsave:byte;

PROCEDURE enableHWA_s3vision(col,init,bpl,bpp,pix:longint);
VAR mode:byte;
BEGIN
  ASM
    MOV DX,03D4h
    MOV AL,40h
    OUT DX,AL
    INC DX
    IN AL,DX
    OR AL,01h
    OUT DX,AL
  END;

  mode:=0;
  CASE pix OF
    640:mode:=mode OR $40;
    800:mode:=mode OR $80;
   1024:mode:=mode OR $00;
   1152:mode:=mode OR $01;
   1280:mode:=mode OR $80;
   1600:mode:=mode OR $C1;
  END;
  CASE bpp OF
    1:mode:=mode OR $00;
    2:mode:=mode OR $10;
    4:mode:=mode OR $30;
  END;
  ASM
    MOV DX,03D4h
    MOV AL,50h
    MOV AH,mode
    OUT DX,AX

    MOV AX,420Fh
    MOV DX,042E8h
    OUT DX,AX

    MOV AX,0008h
    MOV DX,046E8h
    OUT DX,AX

    MOV AX,0FFFFh
    MOV DX,0AAE8h
    OUT DX,AX
    MOV AX,0E010h
    MOV DX,0BEE8h
    OUT DX,AX
    MOV AX,0FFFFh
    MOV DX,0AAE8h
    OUT DX,AX
    MOV AX,0E000h
    MOV DX,0BEE8h
    OUT DX,AX

    MOV AX,0
    MOV DX,0AEE8h
    OUT DX,AX
    MOV AX,0E010h
    MOV DX,0BEE8h
    OUT DX,AX
    MOV AX,0
    MOV DX,0AEE8h
    OUT DX,AX
    MOV AX,0E000h
    MOV DX,0BEE8h
    OUT DX,AX

    MOV AX,0A0C0h
    MOV DX,0BEE8h
    OUT DX,AX

    MOV AX,0027h
    MOV DX,0BAE8h
    OUT DX,AX
  END;
END;

PROCEDURE enableHWA_s3virge(col,init,bpl,bpp,pix:longint);assembler;
ASM
  MOV DX,03D4h
  MOV AL,66h
  OUT DX,AL
  INC DX
  IN AL,DX
  MOV crtc66hsave,AL
  OR AL,88h
  OUT DX,AL
  MOV EDI,reg_mmio
  MOV DWORD PTR [EDI+0ACE8h],0FFFFFFFFh
  MOV DWORD PTR [EDI+0ACECh],0FFFFFFFFh
END;

PROCEDURE disableHWA_s3vision;assembler;
ASM
  MOV DX,03D4h
  MOV AL,40h
  OUT DX,AL
  INC DX
  IN AL,DX
  AND AL,0FEh
  OUT DX,AL
END;

PROCEDURE disableHWA_s3virge;assembler;
ASM
  MOV DX,03D4h
  MOV AL,66h
  OUT DX,AL
  INC DX
  MOV AL,crtc66hsave
  OUT DX,AL
END;

PROCEDURE graphwinhw_s3vision(x1,y1,x2,y2:longint);assembler;
ASM
  MOV DX,0BEE8h

  MOV AX,WORD PTR y1
  ADD AX,WORD PTR frontdrawoffsety
  AND AX,0FFFh
  OR AX,1000h
  OUT DX,AX

  MOV AX,WORD PTR x1
  AND AX,0FFFh
  OR AX,2000h
  OUT DX,AX

  MOV AX,WORD PTR y2
  ADD AX,WORD PTR frontdrawoffsety
  AND AX,0FFFh
  OR AX,3000h
  OUT DX,AX

  MOV AX,WORD PTR x2
  AND AX,0FFFh
  OR AX,4000h
  OUT DX,AX
END;

PROCEDURE graphwinhw_s3virge(x1,y1,x2,y2:longint);assembler;
ASM
  MOV EDI,reg_mmio
  MOV EAX,drawoffset
  MOV [EDI+0A4D4h],EAX
  MOV [EDI+0A4D8h],EAX
  MOV AX,WORD PTR x1
  SHL EAX,16
  MOV AX,WORD PTR x2
  MOV [EDI+0A4DCh],EAX
  MOV AX,WORD PTR y1
  SHL EAX,16
  MOV AX,WORD PTR y2
  MOV [EDI+0A4E0h],EAX
END;

PROCEDURE setpatternHW_s3vision(pattern:ppattern);
VAR x,y:longint;
    p:dword;
BEGIN
{  patternmemoffs:=0; }
  p:=patternmemoffs;
  IF LFBenabled THEN
    BEGIN
      FOR y:=0 TO 7 DO
        BEGIN
          fillchar(pointer(LFBbase+p)^,bytperpix*8,0);
          FOR x:=0 TO 7 DO
            byte(pointer(LFBbase+p+dword(x)*bytperpix)^):=(pattern^[y] SHR x) AND 1;
          {  byte(pointer(LFBbase+p+x*bytperpix)^):=$FF; }
          inc(p,bytperline);
        END;
    END
  ELSE
    BEGIN
      FOR y:=0 TO 7 DO
        BEGIN
          ASM
            MOV EDX,p
            SHR EDX,16
            CALL bankswitch
          END;
          fillchar(pointer(VGAbase+(p AND $FFFF))^,bytperpix*8,0);
          FOR x:=0 TO 7 DO
            byte(pointer(VGAbase+((p+dword(x)*bytperpix) AND $FFFF))^):=(pattern^[y] SHR x) AND 1;
          inc(p,bytperline);
        END;
    END;
END;

PROCEDURE setpatternHW_s3virge(pattern:ppattern);assembler;
ASM
END;

PROCEDURE setpattern16_s3vision;assembler;
ASM
  MOV EDX,0
  MOV EAX,patternmemoffs
  DIV bytperline
  MOV CX,AX
  MOV EAX,EDX
  MOV EDX,0
  DIV bytperpix
  MOV BX,AX

  MOV AX,BX
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,CX
  MOV DX,082E8h
  OUT DX,AX

  MOV AX,0063h
  MOV DX,0B6E8h
  OUT DX,AX

  MOV AX,0001h
  MOV DX,0AEE8h
  OUT DX,AX
END;

PROCEDURE setpattern32_s3vision;assembler;
ASM
  MOV EDX,0
  MOV EAX,patternmemoffs
  DIV bytperline
  MOV CX,AX
  MOV EAX,EDX
  MOV EDX,0
  DIV bytperpix
  MOV BX,AX

  MOV AX,BX
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,CX
  MOV DX,082E8h
  OUT DX,AX

  MOV AX,0063h
  MOV DX,0B6E8h
  OUT DX,AX

  MOV AX,0001h
  MOV DX,0AEE8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0000h
  MOV DX,0AEE8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
END;

{----------------------------- line ---------------------------------------}

PROCEDURE line8n16_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  SUB BX,DI
  JNS @line16_w1
  NEG BX
@line16_w1:
  SUB CX,SI
  JNS @line16_w2
  NEG CX
@line16_w2:
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  CMP BX,CX
  PUSHF
  JLE @line16_w3
  XCHG BX,CX
@line16_w3:
{ BX=dminor, CX=dmajor}
  MOV AX,BX
  SHL AX,1
  MOV DX,08AE8h
  OUT DX,AX
  MOV AX,BX
  SHL AX,1
  MOV DX,CX
  SHL DX,1
  SUB AX,DX
  MOV DX,08EE8h
  OUT DX,AX
  MOV AX,BX
  SHL AX,1
  SUB AX,CX
  MOV DX,092E8h
  OUT DX,AX
  MOV AX,CX
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,2011h
  CMP DI,WORD PTR x2
  JG @line16_w4
  OR AX,0020h
@line16_w4:
  POPF
  JG @line16_w5
  OR AX,0040h
@line16_w5:
  CMP SI,WORD PTR y2
  JG @line16_w6
  OR AX,0080h
@line16_w6:
  MOV DX,09AE8h
  OUT DX,AX
END;

PROCEDURE line32_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  SUB BX,DI
  JNS @line32_w1
  NEG BX
@line32_w1:
  SUB CX,SI
  JNS @line32_w2
  NEG CX
@line32_w2:
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  SHR EAX,16
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  CMP BX,CX
  PUSHF
  JLE @line32_w3
  XCHG BX,CX
@line32_w3:
{ BX=dminor, CX=dmajor }
  MOV AX,BX
  SHL AX,1
  MOV DX,08AE8h
  OUT DX,AX
  MOV AX,BX
  SHL AX,1
  MOV DX,CX
  SHL DX,1
  SUB AX,DX
{  CALL int13bit }
  MOV DX,08EE8h
  OUT DX,AX
  MOV AX,BX
  SHL AX,1
  SUB AX,CX
{  CALL int13bit }
  MOV DX,092E8h
  OUT DX,AX
  MOV AX,CX
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,2011h
  CMP DI,WORD PTR x2
  JG @line32_w4
  OR AX,0020h
@line32_w4:
  POPF
  JG @line32_w5
  OR AX,0040h
@line32_w5:
  CMP SI,WORD PTR y2
  JG @line32_w6
  OR AX,0080h
@line32_w6:
  MOV DX,09AE8h
  OUT DX,AX
END;

PROCEDURE line8_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV EDI,reg_mmio
  MOV ECX,x1
  MOV EBX,y1
  MOV EAX,x2
  MOV EDX,y2
  CMP EBX,EDX
  JLE @lin8_s3v_w1
  XCHG ECX,EAX
  XCHG EBX,EDX
@lin8_s3v_w1:
  MOV [EDI+0A978h],EDX
  MOV SI,AX
  SHL ESI,16
  MOV SI,CX
  MOV [EDI+0A96Ch],ESI
  SUB EDX,EBX
{  INC EDX }
  SUB ECX,EAX
  JS @lin8_w2
  OR EDX,80000000h
@lin8_w2:
  INC EDX
  MOV [EDI+0A97Ch],EDX
{  MOV DWORD PTR GS:[0A4D8h],00000000h }
  MOV EAX,f
  MOV [EDI+0A4F4h],EAX
  MOV EAX,bytperline
  SHL EAX,16
  MOV [EDI+0A4E4h],EAX
  MOV EAX,ECX
{  INC EAX }
  SAL EAX,20
  MOV EBX,EDX
  CDQ
  IDIV EBX
  MOV [EDI+0A970h],EAX
  MOV SI,0
  SHL ESI,4
  ADD EAX,ESI
  MOV [EDI+0A974h],EAX
  MOV DWORD PTR [EDI+0A500h],19E00122h
@lin8_s3v_busy:
  TEST WORD PTR [EDI+08504h],2000h
  JZ @lin8_s3v_busy
END;

PROCEDURE line16_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV EDI,reg_mmio
  MOV ECX,x1
  MOV EBX,y1
  MOV EAX,x2
  MOV EDX,y2
  CMP EBX,EDX
  JLE @lin16_s3v_w1
  XCHG ECX,EAX
  XCHG EBX,EDX
@lin16_s3v_w1:
  MOV [EDI+0A978h],EDX
  MOV SI,AX
  SHL ESI,16
  MOV SI,CX
  MOV [EDI+0A96Ch],ESI
  SUB EDX,EBX
  SUB ECX,EAX
  JS @lin16_w2
  OR EDX,80000000h
@lin16_w2:
  INC DX
  MOV [EDI+0A97Ch],EDX
{  MOV DWORD PTR [EDI+0A4D8h],00000000h }
  MOV EAX,f
  MOV [EDI+0A4F4h],EAX
  MOV EAX,bytperline
  SHL EAX,16
  MOV [EDI+0A4E4h],EAX
  MOV EAX,ECX
  INC EAX
  SAL EAX,20
  MOV EBX,EDX
  CDQ
  IDIV EBX
  MOV [EDI+0A970h],EAX
  MOV SI,0
  SHL ESI,4
  ADD EAX,ESI
  MOV [EDI+0A974h],EAX
  MOV DWORD PTR [EDI+0A500h],19E00126h
@lin16_s3v_busy:
  TEST WORD PTR [EDI+08504h],2000h
  JZ @lin16_s3v_busy
END;

PROCEDURE line32_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV EDI,reg_mmio
  MOV ECX,x1
  MOV EBX,y1
  MOV EAX,x2
  MOV EDX,y2
  CMP EBX,EDX
  JLE @lin32_s3v_w1
  XCHG ECX,EAX
  XCHG EBX,EDX
@lin32_s3v_w1:
  MOV [EDI+0A978h],EDX
  MOV SI,AX
  SHL ESI,16
  MOV SI,CX
  MOV [EDI+0A96Ch],ESI
  SUB EDX,EBX
  SUB ECX,EAX
  JS @lin32_w2
  OR EDX,80000000h
@lin32_w2:
  INC DX
  MOV [EDI+0A97Ch],EDX
{  MOV DWORD PTR [EDI+0A4D8h],00000000h }
  MOV EAX,f
  MOV [EDI+0A4F4h],EAX
  MOV EAX,bytperline
  SHL EAX,16
  MOV [EDI+0A4E4h],EAX
  MOV EAX,ECX
  INC EAX
  SAL EAX,20
  MOV EBX,EDX
  CDQ
  IDIV EBX
  MOV [EDI+0A970h],EAX
  MOV SI,0
  SHL ESI,4
  ADD EAX,ESI
  MOV [EDI+0A974h],EAX
  MOV DWORD PTR [EDI+0A500h],19E0012Dh
@lin32_s3v_busy:
  TEST WORD PTR [EDI+08504h],2000h
  JZ @lin32_s3v_busy
END;

{----------------------------- lineXOR --------------------------------------}

PROCEDURE linexor8n16_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  SUB BX,DI
  JNS @line16_w1
  NEG BX
@line16_w1:
  SUB CX,SI
  JNS @line16_w2
  NEG CX
@line16_w2:
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  CMP BX,CX
  PUSHF
  JLE @line16_w3
  XCHG BX,CX
@line16_w3:
{ BX=dminor, CX=dmajor}
  MOV AX,BX
  SHL AX,1
  MOV DX,08AE8h
  OUT DX,AX
  MOV AX,BX
  SHL AX,1
  MOV DX,CX
  SHL DX,1
  SUB AX,DX
  MOV DX,08EE8h
  OUT DX,AX
  MOV AX,BX
  SHL AX,1
  SUB AX,CX
  MOV DX,092E8h
  OUT DX,AX
  MOV AX,CX
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,0025h
  MOV DX,0BAE8h
  OUT DX,AX
  MOV AX,2011h
  CMP DI,WORD PTR x2
  JG @line16_w4
  OR AX,0020h
@line16_w4:
  POPF
  JG @line16_w5
  OR AX,0040h
@line16_w5:
  CMP SI,WORD PTR y2
  JG @line16_w6
  OR AX,0080h
@line16_w6:
  MOV DX,09AE8h
  OUT DX,AX
  MOV AX,0027h
  MOV DX,0BAE8h
  OUT DX,AX
END;

PROCEDURE linexor32_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  SUB BX,DI
  JNS @line32_w1
  NEG BX
@line32_w1:
  SUB CX,SI
  JNS @line32_w2
  NEG CX
@line32_w2:
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  SHR EAX,16
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  CMP BX,CX
  PUSHF
  JLE @line32_w3
  XCHG BX,CX
@line32_w3:
{ BX=dminor, CX=dmajor }
  MOV AX,BX
  SHL AX,1
  MOV DX,08AE8h
  OUT DX,AX
  MOV AX,BX
  SHL AX,1
  MOV DX,CX
  SHL DX,1
  SUB AX,DX
  MOV DX,08EE8h
  OUT DX,AX
  MOV AX,BX
  SHL AX,1
  SUB AX,CX
  MOV DX,092E8h
  OUT DX,AX
  MOV AX,CX
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,0025h
  MOV DX,0BAE8h
  OUT DX,AX
  MOV AX,2011h
  CMP DI,WORD PTR x2
  JG @line32_w4
  OR AX,0020h
@line32_w4:
  POPF
  JG @line32_w5
  OR AX,0040h
@line32_w5:
  CMP SI,WORD PTR y2
  JG @line32_w6
  OR AX,0080h
@line32_w6:
  MOV DX,09AE8h
  OUT DX,AX
  MOV AX,0027h
  MOV DX,0BAE8h
  OUT DX,AX
END;

{-------------------------------- lineH -------------------------------------}

PROCEDURE lineh8n16_s3vision(x1,x2,y,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @lineh16_w1
  XCHG DI,BX
@lineh16_w1:
  SUB BX,DI
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,0000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,40B1h
  MOV DX,09AE8h
  OUT DX,AX
END;

PROCEDURE lineh32_s3vision(x1,x2,y,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @lineh32_w1
  XCHG DI,BX
@lineh32_w1:
  SUB BX,DI
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  SHR EAX,16
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,0000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,40B1h
  MOV DX,09AE8h
  OUT DX,AX
END;
{
PROCEDURE lineH8_s3virge(x1,x2,y,f:longint);assembler;
ASM
END;

PROCEDURE lineH16_s3virge(x1,x2,y,f:longint);assembler;
ASM
END;

PROCEDURE lineH32_s3virge(x1,x2,y,f:longint);assembler;
ASM
END;
}
{-------------------------------- lineH_pattern -----------------------------}

PROCEDURE lineh8n16_pattern_s3vision(x1,x2,y,f:longint);assembler;
ASM
  CALL setpattern16_s3vision
  MOV DI,WORD PTR x1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @lineh16_w1
  XCHG DI,BX
@lineh16_w1:
  SUB BX,DI
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,08EE8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,08AE8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,0000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0E0B3h
  MOV DX,09AE8h
  OUT DX,AX
  MOV AX,0000h
  MOV DX,0AEE8h
  OUT DX,AX
END;

PROCEDURE lineh32_pattern_s3vision(x1,x2,y,f:longint);assembler;
ASM
  CALL setpattern32_s3vision
  MOV DI,WORD PTR x1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @lineh32_w1
  XCHG DI,BX
@lineh32_w1:
  SUB BX,DI
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  SHR EAX,16
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,08EE8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,08AE8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,0000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0E0B3h
  MOV DX,09AE8h
  OUT DX,AX
  MOV AX,0
  MOV DX,0AEE8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0
  MOV DX,0AEE8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
END;
{
PROCEDURE lineH8_pattern_s3virge(x1,x2,y,f:longint);assembler;
ASM
END;

PROCEDURE lineH16_pattern_s3virge(x1,x2,y,f:longint);assembler;
ASM
END;

PROCEDURE lineH32_pattern_s3virge(x1,x2,y,f:longint);assembler;
ASM
END;
}
{-------------------------------- bar -------------------------------------}

PROCEDURE bar8n16_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @bar16_w1
  XCHG DI,BX
@bar16_w1:
  CMP SI,CX
  JLE @bar16_w2
  XCHG SI,CX
@bar16_w2:
  SUB BX,DI
  SUB CX,SI
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,040B1h
  MOV DX,09AE8h
  OUT DX,AX
@bar16_loop:
  IN AX,DX
  TEST AX,0200h
  JNZ @bar16_loop
END;

PROCEDURE bar32_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @bar32_w1
  XCHG DI,BX
@bar32_w1:
  CMP SI,CX
  JLE @bar32_w2
  XCHG SI,CX
@bar32_w2:
  SUB BX,DI
  SUB CX,SI
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  SHR EAX,16
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,40B1h
  MOV DX,09AE8h
  OUT DX,AX
@bar32_loop:
  IN AX,DX
  TEST AX,0200h
  JNZ @bar32_loop
END;
{
PROCEDURE bar8_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;

PROCEDURE bar16_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;

PROCEDURE bar32_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;
}
{-------------------------------- bar_pattern -------------------------------}

PROCEDURE bar8n16_pattern_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  CALL setpattern16_s3vision
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @bar16_w1
  XCHG DI,BX
@bar16_w1:
  CMP SI,CX
  JLE @bar16_w2
  XCHG SI,CX
@bar16_w2:
  SUB BX,DI
  SUB CX,SI
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,08EE8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,08AE8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0E0B3h
  MOV DX,09AE8h
  OUT DX,AX
@bar16_loop:
  IN AX,DX
  TEST AX,0200h
  JNZ @bar16_loop
  MOV AX,0000h
  MOV DX,0AEE8h
  OUT DX,AX
END;

PROCEDURE bar32_pattern_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  CALL setpattern32_s3vision
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @bar32_w1
  XCHG DI,BX
@bar32_w1:
  CMP SI,CX
  JLE @bar32_w2
  XCHG SI,CX
@bar32_w2:
  SUB BX,DI
  SUB CX,SI
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  SHR EAX,16
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0E0B3h
  MOV DX,09AE8h
  OUT DX,AX
@bar32_loop:
  IN AX,DX
  TEST AX,0200h
  JNZ @bar32_loop
  MOV AX,0
  MOV DX,0AEE8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0
  MOV DX,0AEE8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
END;
{
PROCEDURE bar8_pattern_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;

PROCEDURE bar16_pattern_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;

PROCEDURE bar32_pattern_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;
}
{-------------------------------- barXOR ----------------------------------}

PROCEDURE barXOR8n16_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @barXOR16_w1
  XCHG DI,BX
@barXOR16_w1:
  CMP SI,CX
  JLE @barXOR16_w2
  XCHG SI,CX
@barXOR16_w2:
  SUB BX,DI
  SUB CX,SI
  MOV AX,0025h
  MOV DX,0BAE8h
  OUT DX,AX
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,40B1h
  MOV DX,09AE8h
  OUT DX,AX
@barXOR16_loop:
  IN AX,DX
  TEST AX,0200h
  JNZ @barXOR16_loop
  MOV AX,0027h
  MOV DX,0BAE8h
  OUT DX,AX
END;

PROCEDURE barXOR32_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @barXOR32_w1
  XCHG DI,BX
@barXOR32_w1:
  CMP SI,CX
  JLE @barXOR32_w2
  XCHG SI,CX
@barXOR32_w2:
  SUB BX,DI
  SUB CX,SI
  MOV AX,0025h
  MOV DX,0BAE8h
  OUT DX,AX
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  SHR EAX,16
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,086E8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,082E8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,40B1h
  MOV DX,09AE8h
  OUT DX,AX
@barXOR32_loop:
  IN AX,DX
  TEST AX,0200h
  JNZ @barXOR32_loop
  MOV AX,0027h
  MOV DX,0BAE8h
  OUT DX,AX
END;
{
PROCEDURE barXOR8_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;

PROCEDURE barXOR16_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;

PROCEDURE barXOR32_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;
}
{-------------------------------- barXOR_pattern ----------------------------}

PROCEDURE barXOR8n16_pattern_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  CALL setpattern16_s3vision
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @barXOR16_w1
  XCHG DI,BX
@barXOR16_w1:
  CMP SI,CX
  JLE @barXOR16_w2
  XCHG SI,CX
@barXOR16_w2:
  SUB BX,DI
  SUB CX,SI
  MOV AX,0025h
  MOV DX,0BAE8h
  OUT DX,AX
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,08EE8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,08AE8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0E0B3h
  MOV DX,09AE8h
  OUT DX,AX
@barXOR16_loop:
  IN AX,DX
  TEST AX,0200h
  JNZ @barXOR16_loop
  MOV AX,0027h
  MOV DX,0BAE8h
  OUT DX,AX
  MOV AX,0000h
  MOV DX,0AEE8h
  OUT DX,AX
END;

PROCEDURE barXOR32_pattern_s3vision(x1,y1,x2,y2,f:longint);assembler;
ASM
  CALL setpattern32_s3vision
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  ADD SI,WORD PTR frontdrawoffsety
  ADD CX,WORD PTR frontdrawoffsety
  CMP DI,BX
  JLE @barXOR32_w1
  XCHG DI,BX
@barXOR32_w1:
  CMP SI,CX
  JLE @barXOR32_w2
  XCHG SI,CX
@barXOR32_w2:
  SUB BX,DI
  SUB CX,SI
  MOV AX,0025h
  MOV DX,0BAE8h
  OUT DX,AX
  MOV EAX,f
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  SHR EAX,16
  MOV DX,0A6E8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,DI
  AND AX,0FFFh
  MOV DX,08EE8h
  OUT DX,AX
  MOV AX,SI
  AND AX,0FFFh
  MOV DX,08AE8h
  OUT DX,AX
  MOV AX,BX
  AND AX,0FFFh
  MOV DX,096E8h
  OUT DX,AX
  MOV AX,CX
  AND AX,0FFFh
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0E0B1h
  MOV DX,09AE8h
  OUT DX,AX
@barXOR32_loop:
  IN AX,DX
  TEST AX,0200h
  JNZ @barXOR32_loop
  MOV AX,0027h
  MOV DX,0BAE8h
  OUT DX,AX
  MOV AX,0
  MOV DX,0AEE8h
  OUT DX,AX
  MOV AX,0E010h
  MOV DX,0BEE8h
  OUT DX,AX
  MOV AX,0
  MOV DX,0AEE8h
  OUT DX,AX
  MOV AX,0E000h
  MOV DX,0BEE8h
  OUT DX,AX
END;
{
PROCEDURE barXOR8_pattern_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;

PROCEDURE barXOR16_pattern_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;

PROCEDURE barXOR32_pattern_s3virge(x1,y1,x2,y2,f:longint);assembler;
ASM
END;
}
{------------------------- moverect ---------------------------------------}

PROCEDURE moverect_s3vision(x1,y1,x2,y2,x,y:longint);assembler;
VAR draw:word;
ASM
  MOV DI,WORD PTR x1
  MOV SI,WORD PTR y1
  MOV BX,WORD PTR x2
  MOV CX,WORD PTR y2
  MOV draw,0C0B1h

  MOV AX,WORD PTR frontdrawoffsety
  ADD SI,AX
  ADD CX,AX
  ADD WORD PTR y,AX

  CMP DI,BX
  JLE @mr_w1
  XCHG DI,BX
@mr_w1:

  CMP SI,CX
  JLE @mr_w2
  XCHG SI,CX
@mr_w2:

  SUB BX,DI
  SUB CX,SI

  MOV AX,DI
  CMP AX,WORD PTR x
  JGE @mr_w3
  ADD AX,BX
  ADD WORD PTR x,BX
  AND draw,0FFDFh
@mr_w3:
{  AND AX,0FFFh}
  MOV DX,086E8h
  OUT DX,AX

  MOV AX,SI
  CMP AX,WORD PTR y
  JGE @mr_w4
  ADD AX,CX
  ADD WORD PTR y,CX
  AND draw,0FF7Fh
@mr_w4:
{  AND AX,0FFFh}
  MOV DX,082E8h
  OUT DX,AX

  MOV AX,WORD PTR x
{  AND AX,0FFFh}
  MOV DX,08EE8h
  OUT DX,AX

  MOV AX,WORD PTR y
{  AND AX,0FFFh }
  MOV DX,08AE8h
  OUT DX,AX

  MOV AX,BX
{  AND AX,0FFFh }

  MOV DX,096E8h
  OUT DX,AX

  MOV AX,CX
{  AND AX,0FFFh }
  MOV DX,0BEE8h
  OUT DX,AX

  MOV AX,00067h
  MOV DX,0BAE8h
  OUT DX,AX

  MOV AX,draw
  MOV DX,09AE8h
  OUT DX,AX

  MOV AX,0027h
  MOV DX,0BAE8h
  OUT DX,AX

  MOV DX,09AE8h
@tr16_s3_loop:
  IN AX,DX
  TEST AX,0200h
  JNZ @tr16_s3_loop
END;

PROCEDURE moverect8_s3virge(x1,y1,x2,y2,x,y:longint);assembler;
ASM
  MOV EAX,x1
  MOV EBX,y1
  MOV ECX,x2
  MOV EDX,y2
  CMP EAX,ECX
  JLE @mr8_s3v_w1
  XCHG EAX,ECX
@mr8_s3v_w1:
  CMP EBX,EDX
  JLE @mr8_s3v_w2
  XCHG EBX,EDX
@mr8_s3v_w2:
  SUB ECX,EAX
  SUB EDX,EBX
  MOV ESI,x
  MOV EDI,00000000h
  CMP SI,AX
  JG @mr8_s3v_xup
  OR EDI,02000000h
  JMP @mr8_s3v_w3x
@mr8_s3v_xup:
  ADD AX,CX
  ADD SI,CX
@mr8_s3v_w3x:
  SHL ESI,16
  MOV SI,WORD PTR y
  CMP SI,BX
  JG @mr8_s3v_yup
  OR EDI,04000000h
  JMP @mr8_s3v_w3y
@mr8_s3v_yup:
  ADD BX,DX
  ADD SI,DX
@mr8_s3v_w3y:
  SHL EAX,16
  MOV AX,BX
  SHL ECX,16
  INC DX
  MOV CX,DX
  MOV EBX,reg_mmio
{  MOV DWORD PTR GS:[0A4D4h],0
  MOV DWORD PTR GS:[0A4D8h],0 }
  MOV [EBX+0A504h],ECX
  MOV [EBX+0A508h],EAX
  MOV [EBX+0A50Ch],ESI
  MOV AX,WORD PTR bytperline
  SHL EAX,16
  MOV AX,WORD PTR bytperline
  MOV [EBX+0A4E4h],EAX
  OR EDI,01980022h
  MOV [EBX+0A500h],EDI
@mr8_s3v_busy:
  TEST WORD PTR [EBX+08504h],2000h
  JZ @mr8_s3v_busy
END;

PROCEDURE moverect16_s3virge(x1,y1,x2,y2,x,y:longint);assembler;
ASM
  MOV EAX,x1
  MOV EBX,y1
  MOV ECX,x2
  MOV EDX,y2
  CMP EAX,ECX
  JLE @mr16_s3v_w1
  XCHG EAX,ECX
@mr16_s3v_w1:
  CMP EBX,EDX
  JLE @mr16_s3v_w2
  XCHG EBX,EDX
@mr16_s3v_w2:
  SUB ECX,EAX
  SUB EDX,EBX
  MOV ESI,x
  MOV EDI,00000000h
  CMP SI,AX
  JG @mr16_s3v_xup
  OR EDI,02000000h
  JMP @mr16_s3v_w3x
@mr16_s3v_xup:
  ADD AX,CX
  ADD SI,CX
@mr16_s3v_w3x:
  SHL ESI,16
  MOV SI,WORD PTR y
  CMP SI,BX
  JG @mr16_s3v_yup
  OR EDI,04000000h
  JMP @mr16_s3v_w3y
@mr16_s3v_yup:
  ADD BX,DX
  ADD SI,DX
@mr16_s3v_w3y:
  SHL EAX,16
  MOV AX,BX
  SHL ECX,16
  INC DX
  MOV CX,DX
  MOV EBX,reg_mmio
{  MOV DWORD PTR GS:[0A4D4h],0
  MOV DWORD PTR GS:[0A4D8h],0 }
  MOV [EBX+0A504h],ECX
  MOV [EBX+0A508h],EAX
  MOV [EBX+0A50Ch],ESI
  MOV AX,WORD PTR bytperline
  SHL EAX,16
  MOV AX,WORD PTR bytperline
  MOV [EBX+0A4E4h],EAX
  OR EDI,01980026h
  MOV [EBX+0A500h],EDI
@mr16_s3v_busy:
  TEST WORD PTR [EBX+08504h],2000h
  JZ @mr16_s3v_busy
END;

PROCEDURE moverect32_s3virge(x1,y1,x2,y2,x,y:longint);assembler;
ASM
  MOV EAX,x1
  MOV EBX,y1
  MOV ECX,x2
  MOV EDX,y2
  CMP EAX,ECX
  JLE @mr32_s3v_w1
  XCHG EAX,ECX
@mr32_s3v_w1:
  CMP EBX,EDX
  JLE @mr32_s3v_w2
  XCHG EBX,EDX
@mr32_s3v_w2:
  SUB ECX,EAX
  SUB EDX,EBX
  MOV ESI,x
  MOV EDI,00000000h
  CMP SI,AX
  JG @mr32_s3v_xup
  OR EDI,02000000h
  JMP @mr32_s3v_w3x
@mr32_s3v_xup:
  ADD AX,CX
  ADD SI,CX
@mr32_s3v_w3x:
  SHL ESI,16
  MOV SI,WORD PTR y
  CMP SI,BX
  JG @mr32_s3v_yup
  OR EDI,04000000h
  JMP @mr32_s3v_w3y
@mr32_s3v_yup:
  ADD BX,DX
  ADD SI,DX
@mr32_s3v_w3y:
  SHL EAX,16
  MOV AX,BX
  SHL ECX,16
  INC DX
  MOV CX,DX
  MOV EBX,reg_mmio
{  MOV DWORD PTR GS:[0A4D4h],0
  MOV DWORD PTR GS:[0A4D8h],0 }
  MOV [EBX+0A504h],ECX
  MOV [EBX+0A508h],EAX
  MOV [EBX+0A50Ch],ESI
  MOV AX,WORD PTR bytperline
  SHL EAX,16
  MOV AX,WORD PTR bytperline
  MOV [EBX+0A4E4h],EAX
  OR EDI,0198002Dh
  MOV [EBX+0A500h],EDI
@mr32_s3v_busy:
  TEST WORD PTR [EBX+08504h],2000h
  JZ @mr32_s3v_busy
END;

{============================================================================}

PROCEDURE setreslist_s3vision(mem,chipid:dword;var modelist:PModeEntry);
BEGIN
  AddModeToList(modelist,$69,  640, 480,1,  640, $50,$4050,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$4E,  640, 480,2, 1280, $A0,$5050,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(modelist,$4F,  640, 480,2, 1280, $A0,$5050,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(modelist,$50,  640, 480,3, 1920, $F0,$4050,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_col24);
  AddModeToList(modelist,$70,  640, 480,4, 2560,$140,$7050,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(modelist,$6B,  800, 600,1,  800, $64,$8050,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$74,  800, 600,2, 1600, $C8,$9050,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(modelist,$7B,  800, 600,4, 3200,$190,$B050,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(modelist,$6D, 1024, 768,1, 1024, $80,$0050,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$77, 1024, 768,2, 2048,$100,$1050,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(modelist,$78, 1024, 768,4, 4096,$200,$3050,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(modelist,$3F, 1152, 864,1, 1152, $90,$0150,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$72, 1280,1024,1, 1280, $A0,$C050,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$79, 1280,1024,2, 2560,$140,$D050,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
END;

PROCEDURE setreslist_s3trio(mem,chipid:dword;var modelist:PModeEntry);
BEGIN
  AddModeToList(modelist,$65,  320, 200,2,  640, $50,0,    0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_col15);
  AddModeToList(modelist,$66,  320, 200,2,  640, $50,0,    0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_col16);
  AddModeToList(modelist,$67,  320, 200,4, 1280, $A0,0,    0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_col32);
  AddModeToList(modelist,$68,  640, 400,1,  640, $50,$4050,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$52,  640, 400,4, 2560,$140,$7050,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(modelist,$69,  640, 480,1,  640, $50,$4050,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$70,  640, 480,2, 1280, $A0,$5050,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(modelist,$71,  640, 480,2, 1280, $A0,$5050,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(modelist,$51,  640, 480,3, 1920, $F0,$4050,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_col24);
  AddModeToList(modelist,$72,  640, 480,4, 2560,$140,$7050,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(modelist,$6B,  800, 600,1,  800, $64,$8050,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$73,  800, 600,2, 1600, $C8,$9050,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(modelist,$74,  800, 600,2, 1600, $C8,$9050,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(modelist,$7B,  800, 600,4, 3200,$190,$B050,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(modelist,$6D, 1024, 768,1, 1024, $80,$0050,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$76, 1024, 768,2, 2048,$100,$1050,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(modelist,$77, 1024, 768,2, 2048,$100,$1050,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(modelist,$78, 1024, 768,4, 4096,$200,$3050,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(modelist,$4E, 1152, 864,1, 1152, $90,$0150,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$6F, 1280,1024,1, 1280, $A0,$C050,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$79, 1280,1024,2, 2560,$140,$D050,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(modelist,$7A, 1280,1024,2, 2560,$140,$D050,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$7B, 1280,1024,3, 3840,$1F0,0,    0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_col15);
  AddModeToList(modelist,$7C, 1600,1200,1, 1600, $64,$8150,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(modelist,$7D, 1600,1200,2, 3200, $C8,$9150,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(modelist,$7E, 1600,1200,2, 3200, $C8,$9150,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
END;

PROCEDURE setreslist_s3virge(mem,chipid:dword;var modelist:PModeEntry);
BEGIN
  AddModeToList(CurGraphiX.modelist,$13,  320, 200,1,  320, $28,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$44,  320, 200,2,  640, $50,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$34,  320, 200,2,  640, $50,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(CurGraphiX.modelist,$24,  320, 200,3,  960, $78,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24);
  AddModeToList(CurGraphiX.modelist,$24,  320, 200,4, 1280, $A0,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(CurGraphiX.modelist,$55,  320, 240,1,  320, $28,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$45,  320, 240,2,  640, $50,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$35,  320, 240,2,  640, $50,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(CurGraphiX.modelist,$25,  320, 240,3,  960, $78,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24);
  AddModeToList(CurGraphiX.modelist,$25,  320, 240,4, 1280, $A0,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(CurGraphiX.modelist,$56,  400, 300,1,  400, $32,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$46,  400, 300,2,  800, $64,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$36,  400, 300,2,  800, $64,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(CurGraphiX.modelist,$26,  400, 300,3, 1200, $96,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24);
  AddModeToList(CurGraphiX.modelist,$26,  400, 300,4, 1600, $C8,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(CurGraphiX.modelist,$57,  512, 384,1,  512, $40,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$47,  512, 384,2, 1024, $80,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$37,  512, 384,2, 1024, $80,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(CurGraphiX.modelist,$27,  512, 384,3, 1536, $C0,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24);
  AddModeToList(CurGraphiX.modelist,$27,  512, 384,4, 2048,$100,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(CurGraphiX.modelist,$58,  640, 400,1,  640, $50,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$48,  640, 400,2, 1280, $A0,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$38,  640, 400,2, 1280, $A0,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(CurGraphiX.modelist,$28,  640, 400,3, 1920, $F0,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24);
  AddModeToList(CurGraphiX.modelist,$28,  640, 400,4, 2560,$140,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(CurGraphiX.modelist,$59,  640, 480,1,  640, $50,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$49,  640, 480,2, 1280, $A0,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$39,  640, 480,2, 1280, $A0,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(CurGraphiX.modelist,$29,  640, 480,3, 1920, $F0,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24);
  AddModeToList(CurGraphiX.modelist,$29,  640, 480,4, 2560,$140,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(CurGraphiX.modelist,$5A,  800, 600,1,  800, $64,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$4A,  800, 600,2, 1600, $C8,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$3A,  800, 600,2, 1600, $C8,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(CurGraphiX.modelist,$2A,  800, 600,3, 2400,$12C,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24);
  AddModeToList(CurGraphiX.modelist,$2A,  800, 600,4, 3200,$190,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(CurGraphiX.modelist,$5B, 1024, 768,1, 1024, $80,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$4B, 1024, 768,2, 2048,$100,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$3B, 1024, 768,2, 2048,$100,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(CurGraphiX.modelist,$2B, 1024, 768,3, 3072,$180,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24);
  AddModeToList(CurGraphiX.modelist,$2B, 1024, 768,4, 4096,$200,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(CurGraphiX.modelist,$5C, 1152, 864,1, 1152, $90,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$4C, 1152, 864,2, 2304,$120,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$3C, 1152, 864,2, 2304,$120,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
  AddModeToList(CurGraphiX.modelist,$2C, 1152, 864,3, 3456,$1B0,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24);
  AddModeToList(CurGraphiX.modelist,$2C, 1152, 864,4, 4608,$240,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col32);
  AddModeToList(CurGraphiX.modelist,$5D, 1280,1024,1, 1280, $A0,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
  AddModeToList(CurGraphiX.modelist,$4D, 1280,1024,2, 2560,$140,0,0, 10,5,5,5,0,5, ig_bank+ig_lfb+ig_hwa+ig_col15);
  AddModeToList(CurGraphiX.modelist,$3D, 1280,1024,2, 2560,$140,0,0, 11,5,5,6,0,5, ig_bank+ig_lfb+ig_hwa+ig_col16);
{  AddModeToList(CurGraphiX.modelist,$3D, 1280,1024,3, 3840,$1E0,0,0, 16,8,8,8,0,8, ig_bank+ig_lfb+ig_hwa+ig_col24); }
  AddModeToList(CurGraphiX.modelist,$5E, 1600,1200,1, 1600, $C8,0,0,  5,3,2,3,0,2, ig_bank+ig_lfb+ig_hwa+ig_col8);
END;

{============================================================================}

PROCEDURE sethwaprocs_s3vision(col:longint);
BEGIN
  CASE col OF
    ig_col8:
      BEGIN
        graphwinHW:=@graphwinHW_s3vision;
        setpatternHW:=@setpatternHW_s3vision;
        line:=@line8n16_s3vision;
        lineXOR:=@lineXOR8n16_s3vision;
        lineH:=@lineH8n16_s3vision;
        lineH_solid:=@lineH8n16_s3vision;
        lineH_pattern:=@lineH8n16_pattern_s3vision;
        bar:=@bar8n16_s3vision;
        bar_solid:=@bar8n16_s3vision;
        bar_pattern:=@bar8n16_pattern_s3vision;
        barXOR:=@barXOR8n16_S3vision;
        barXOR_solid:=@barXOR8n16_S3vision;
        barXOR_pattern:=@barXOR8n16_pattern_s3vision;
        moverect:=@moverect_s3vision;
      END;
    ig_col15:
      BEGIN
        graphwinHW:=@graphwinHW_s3vision;
        setpatternHW:=@setpatternHW_s3vision;
        line:=@line8n16_s3vision;
        lineXOR:=@lineXOR8n16_s3vision;
        lineH:=@lineH8n16_s3vision;
        lineH_solid:=@lineH8n16_s3vision;
        lineH_pattern:=@lineH8n16_pattern_s3vision;
        bar:=@bar8n16_s3vision;
        bar_solid:=@bar8n16_s3vision;
        bar_pattern:=@bar8n16_pattern_s3vision;
        barXOR:=@barXOR8n16_S3vision;
        barXOR_solid:=@barXOR8n16_S3vision;
        barXOR_pattern:=@barXOR8n16_pattern_s3vision;
        moverect:=@moverect_s3vision;
      END;
    ig_col16:
      BEGIN
        graphwinHW:=@graphwinHW_s3vision;
        setpatternHW:=@setpatternHW_s3vision;
        line:=@line8n16_s3vision;
        lineXOR:=@lineXOR8n16_s3vision;
        lineH:=@lineH8n16_s3vision;
        lineH_solid:=@lineH8n16_s3vision;
        lineH_pattern:=@lineH8n16_pattern_s3vision;
        bar:=@bar8n16_s3vision;
        bar_solid:=@bar8n16_s3vision;
        bar_pattern:=@bar8n16_pattern_s3vision;
        barXOR:=@barXOR8n16_S3vision;
        barXOR_solid:=@barXOR8n16_S3vision;
        barXOR_pattern:=@barXOR8n16_pattern_s3vision;
        moverect:=@moverect_s3vision;
      END;
    ig_col32:
      BEGIN
        graphwinHW:=@graphwinHW_s3vision;
        setpatternHW:=@setpatternHW_s3vision;
        line:=@line32_s3vision;
        lineXOR:=@lineXOR32_s3vision;
        lineH:=@lineH32_s3vision;
        lineH_solid:=@lineH32_s3vision;
        lineH_pattern:=@lineH32_pattern_s3vision;
        bar:=@bar32_s3vision;
        bar_solid:=@bar32_s3vision;
        bar_pattern:=@bar32_pattern_s3vision;
        barXOR:=@barXOR32_S3vision;
        barXOR_solid:=@barXOR32_S3vision;
        barXOR_pattern:=@barXOR32_pattern_s3vision;
        moverect:=@moverect_s3vision;
      END;
  END;
END;

PROCEDURE sethwaprocs_s3virge(col:longint);
BEGIN
  CASE col OF
    ig_col8:
      BEGIN
        graphwinHW:=@graphwinHW_S3virge;
        setpatternHW:=@setpatternHW_s3virge;
        line:=@line8_S3virge;
     {  lineH:=@lineH8_S3v;
        bar:=@bar8_S3v;
        barXOR:=@barXOR8_S3v; }
        moverect:=@moverect8_S3virge;
      END;
    ig_col15:
      BEGIN
        graphwinHW:=@graphwinHW_S3virge;
        setpatternHW:=@setpatternHW_s3virge;
        line:=@line16_S3virge;
     {  lineH:=@lineH16_S3v;
        bar:=@bar16_S3v;
        barXOR:=@barXOR16_S3v; }
        moverect:=@moverect16_S3virge;
      END;
    ig_col16:
      BEGIN
        graphwinHW:=@graphwinHW_S3virge;
        setpatternHW:=@setpatternHW_s3virge;
        line:=@line16_S3virge;
     {  lineH:=@lineH16_S3v;
        bar:=@bar16_S3v;
        barXOR:=@barXOR16_S3v; }
        moverect:=@moverect16_S3virge;
      END;
    ig_col24:
      BEGIN
        graphwinHW:=@graphwinHW_S3virge;
        setpatternHW:=@setpatternHW_s3virge;
      END;
    ig_col32:
      BEGIN
        graphwinHW:=@graphwinHW_S3virge;
        setpatternHW:=@setpatternHW_s3virge;
        line:=@line32_S3virge;
     {  lineH:=@lineH16_S3v;
        bar:=@bar16_S3v;
        barXOR:=@barXOR16_S3v; }
        moverect:=@moverect32_S3virge;
      END;
  END;
END;

END.
