///////////////////////////////////////////////////////////////////////////////
// ddex3_api.dpr
///////////////////////////////////////////////////////////////////////////////
// Delphi conversion of the ddex3 example contained in Microsoft's DirectX sdk.
// Based on Eric Unger's conversions of the DirectX headers. They're available
// at http://www.delphi-jedi.org. Bug reports to meyerhoff@earthling.net.
///////////////////////////////////////////////////////////////////////////////
// Description:
// Direct Draw example program 3. Adds functionality to example program 2.
// Creates two offscreen surfaces in addition to the primary surface and back
// buffer. Loads a bitmap file into each offscreen surface. Uses BltFast to
// copy the contents of an offscreen surface to the back buffer and then flips
// the buffers and copies the next offscreen surface to the back buffer. Press
// F12 or ESCAPE to exit the program. This program requires at least 1.2 Megs
// of video ram.
///////////////////////////////////////////////////////////////////////////////
program ddex3_api;
{$MODE DELPHI}

//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------
uses
  Windows,

  DirectDraw,
  SysUtils,
  DDUtil,
  DirectDrawFPC;

const
  //---------------------------------------------------------------------------
  // Local definitions
  //---------------------------------------------------------------------------
  NAME  : PChar = 'DDExample3';
  TITLE : PChar = 'Direct Draw Example 3';

  //---------------------------------------------------------------------------
  // Default settings
  //---------------------------------------------------------------------------
  TIMER_ID   = 1;
  TIMER_RATE = 500;

var
  //---------------------------------------------------------------------------
  // Global data
  //---------------------------------------------------------------------------
  g_pDD         : IDirectDraw4;                // DirectDraw object
  g_pDDSPrimary : IDirectDrawSurface4;         // DirectDraw primary surface
  g_pDDSBack    : IDirectDrawSurface4;         // DirectDraw back surface
  g_pDDSOne     : IDirectDrawSurface4;         // Offscreen surface 1
  g_pDDSTwo     : IDirectDrawSurface4;         // Offscreen surface 2
  g_pDDPal      : IDIRECTDRAWPALETTE;          // The primary surface palette
  g_bActive     : Boolean = False;             // Is application active?

  //---------------------------------------------------------------------------
  // Local data
  //---------------------------------------------------------------------------
  // Name of our bitmap resource.
  szBitmap : PChar = 'frntback.bmp';

//-----------------------------------------------------------------------------
// Name: ReleaseAllObjects
// Desc: Finished with all objects we use; release them
//-----------------------------------------------------------------------------
procedure ReleaseAllObjects;
begin
  if Assigned(g_pDD) then
    begin
      if Assigned(g_pDDSBack) then
        begin
          g_pDDSBack := nil;
        end;
      if Assigned(g_pDDSPrimary) then
        begin
          g_pDDSPrimary := nil;
        end;
      if Assigned(g_pDDSOne) then
        begin
          g_pDDSOne := nil;
        end;
      if Assigned(g_pDDSTwo) then
        begin
          g_pDDSTwo := nil;
        end;
      if Assigned(g_pDDPal) then
        begin
          g_pDDPal := nil;
        end;
      g_pDD := nil;
    end;
end;

//-----------------------------------------------------------------------------
// Name: InitFail
// Desc: This function is called if an initialization function fails
//-----------------------------------------------------------------------------
function InitFail(h_Wnd : HWND; hRet : HRESULT; Text : string) : HRESULT;
begin
  ReleaseAllObjects;
  MessageBox(h_Wnd, PChar(Text + ': ' + DDErrorString(hRet)), TITLE, MB_OK);
  DestroyWindow(h_Wnd);
  Result := hRet;
end;

//-----------------------------------------------------------------------------
// Name: InitSurfaces
// Desc: This function reads the bitmap file FRNTBACK.BMP and stores half of it
//       in offscreen surface 1 and the other half in offscreen surface 2.
//-----------------------------------------------------------------------------
function InitSurfaces : Boolean;
var
  hbm : HBITMAP;
begin
  hbm := LoadImage(0, szBitmap, IMAGE_BITMAP, 0, 0, LR_LOADFROMFILE or LR_CREATEDIBSECTION);
  if hbm = 0 then
    begin
      Result := False;
      Exit;
    end;
  DDCopyBitmap(g_pDDSOne, hbm, 0, 0, 640, 480);
  DDCopyBitmap(g_pDDSTwo, hbm, 0, 480, 640, 480);
  DeleteObject(hbm);
  Result := True;
end;

//-----------------------------------------------------------------------------
// Name: RestoreAll()
// Desc: Restore all lost objects
//-----------------------------------------------------------------------------
function RestoreAll : HRESULT;
var
  hRet : HRESULT;
begin
  hRet := IDirectDrawSurface4__Restore(g_pDDSPrimary);
  if hRet = DD_OK then
    begin
      hRet := IDirectDrawSurface4__Restore(g_pDDSOne);
      if hRet = DD_OK then
        begin
          hRet := IDirectDrawSurface4__Restore(g_pDDSTwo);
          if hRet = DD_OK then
            begin
                InitSurfaces();
            end;
        end;
    end;
  Result := hRet;
end;

//-----------------------------------------------------------------------------
// Name: UpdateFrame()
// Desc: Displays the proper image for the page
//-----------------------------------------------------------------------------
var
  phase : Boolean = False;

procedure UpdateFrame(h_Wnd : HWND);
var
  rcRect : TRect;
  pdds : IDirectDrawSurface4;
  hRet : HRESULT;
begin
  rcRect.left := 0;
  rcRect.top := 0;
  rcRect.right := 640;
  rcRect.bottom := 480;

  if phase then
    begin
      pdds := g_pDDSTwo;
      phase := False;
    end
  else
    begin
      pdds := g_pDDSOne;
      phase := True;
    end;
  while True do
    begin
      hRet := IDirectDrawSurface4_BltFast(g_pDDSBack, 0, 0, pdds, @rcRect, 0);
      if hRet = DD_OK then
        Break;
      if hRet = DDERR_SURFACELOST then
        begin
          hRet := RestoreAll;
          if hRet <> DD_OK then
            Break;
        end;
      if hRet <> DDERR_WASSTILLDRAWING then
        Break;
    end;
end;

//-----------------------------------------------------------------------------
// Name: WindowProc
// Desc: The Main Window Procedure
//-----------------------------------------------------------------------------
function WindowProc(h_Wnd: HWND; aMSG: Cardinal; wParam: Cardinal; lParam: Integer) : Integer; stdcall;
var
  hRet : HRESULT;
begin
  case aMSG of
    // Pause if minimized
    WM_ACTIVATE:
      begin
        if HIWORD(wParam) = 0 then
          g_bActive := True
        else
          g_bActive := False;
        Result := 0;
        Exit;
      end;
    // Clean up and close the app
    WM_DESTROY:
      begin
        ReleaseAllObjects;
        PostQuitMessage(0);
        Result := 0;
        Exit;
      end;
    // Handle any non-accelerated key commands
    WM_KEYDOWN:
      begin
        case wParam of
          VK_ESCAPE,
          VK_F12:
            begin
              PostMessage(h_Wnd, WM_CLOSE, 0, 0);
              Result := 0;
              Exit;
            end;
        end;
      end;
    // Turn off the cursor since this is a full-screen app
    WM_SETCURSOR:
      begin
        SetCursor(0);
        Result := 1;
        Exit;
      end;
    // Update and flip surfaces
    WM_TIMER:
      begin
        if g_bActive and (TIMER_ID = wParam) then
          begin
            UpdateFrame(h_Wnd);
            while True do
              begin
                hRet := IDirectDrawSurface4_Flip(g_pDDSPrimary, nil, 0);
                if hRet = DD_OK then
                    Break;
                if hRet = DDERR_SURFACELOST then
                  begin
                    hRet := RestoreAll;
                    if hRet <> DD_OK then
                      Break;
                  end;
                if hRet <> DDERR_WASSTILLDRAWING then
                  Break;
              end;
          end;
      end;
    end;

  Result := DefWindowProc(h_Wnd, aMSG, wParam, lParam);
end;

//-----------------------------------------------------------------------------
// Name: InitApp
// Desc: Do work required for every instance of the application:
//          Create the window, initialize data
//-----------------------------------------------------------------------------
function InitApp(hInst : THANDLE; nCmdShow : Integer) : HRESULT;
var
  h_Wnd : HWND;
  wc : WNDCLASS;
  ddsd : TDDSurfaceDesc2;
  ddscaps : TDDSCaps2;
  hRet : HRESULT;
  pddTemp : IDirectDraw;
begin
  // Set up and register window class
  wc.style := CS_HREDRAW or CS_VREDRAW;
  wc.lpfnWndProc := @WindowProc;
  wc.cbClsExtra := 0;
  wc.cbWndExtra := 0;
  wc.hInstance := hInst;
  wc.hIcon := LoadIcon(hInst, 'MAINICON');
  wc.hCursor := LoadCursor(0, IDC_ARROW);
  wc.hbrBackground := GetStockObject(BLACK_BRUSH);
  wc.lpszMenuName := NAME;
  wc.lpszClassName := NAME;
  RegisterClass(wc);

  // Create a window
  h_Wnd := CreateWindowEx(WS_EX_TOPMOST,
                          NAME,
                          TITLE,
                          WS_POPUP,
                          0,
                          0,
                          GetSystemMetrics(SM_CXSCREEN),
                          GetSystemMetrics(SM_CYSCREEN),
                          0,
                          0,
                          hInst,
                          nil);

  if h_Wnd = 0 then
    begin
      Result := 0;
      Exit;
    end;

  ShowWindow(h_Wnd, nCmdShow);
  UpdateWindow(h_Wnd);
  SetFocus(h_Wnd);

  ///////////////////////////////////////////////////////////////////////////
  // Create the main DirectDraw object
  ///////////////////////////////////////////////////////////////////////////
  hRet := DirectDrawCreate(nil, pDDTemp, nil);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'DirectDrawCreate FAILED');
      Exit;
    end;
  hRet := IDirectDraw_QueryInterface(pDDTemp, IID_IDirectDraw4, g_pDD);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'QueryInterface FAILED');
      Exit;
    end;
  IDirectDraw_Release(pDDTemp);
  pDDTemp := nil;


  // Get exclusive mode
  hRet := IDirectDraw4_SetCooperativeLevel(g_pDD, h_Wnd, DDSCL_EXCLUSIVE or DDSCL_FULLSCREEN);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetCooperativeLevel FAILED');
      Exit;
    end;

  // Set the video mode to 640x480x8
  hRet := IDirectDraw4_SetDisplayMode(g_pDD, 640, 480, 8, 0, 0);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetDisplayMode FAILED');
      Exit;
    end;

  // Create the primary surface with 1 back buffer
  FillChar(ddsd, SizeOf(ddsd), 0);
  ddsd.dwSize := SizeOf(ddsd);
  ddsd.dwFlags := DDSD_CAPS or DDSD_BACKBUFFERCOUNT;
  ddsd.ddsCaps.dwCaps := DDSCAPS_PRIMARYSURFACE or DDSCAPS_FLIP or DDSCAPS_COMPLEX;
  ddsd.dwBackBufferCount := 1;
  hRet := IDirectDraw4_CreateSurface(g_pDD, ddsd, g_pDDSPrimary, nil);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'CreateSurface FAILED');
      Exit;
    end;

  // Get a pointer to the back buffer
  FillChar(ddscaps, SizeOf(ddscaps), 0);
  ddscaps.dwCaps := DDSCAPS_BACKBUFFER;
  hRet := IDirectDrawSurface4_GetAttachedSurface(g_pDDSPrimary, ddscaps, g_pDDSBack);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'GetAttachedSurface FAILED');
      Exit;
    end;

  // Create a offscreen bitmap.
  ddsd.dwFlags := DDSD_CAPS or DDSD_HEIGHT or DDSD_WIDTH;
  ddsd.ddsCaps.dwCaps := DDSCAPS_OFFSCREENPLAIN;
  ddsd.dwHeight := 480;
  ddsd.dwWidth := 640;
  hRet := IDirectDraw4_CreateSurface(g_pDD, ddsd, g_pDDSOne, nil);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'CreateSurface FAILED');
      Exit;
    end;

  // Create another offscreen bitmap.
  hRet := IDirectDraw4_CreateSurface(g_pDD, ddsd, g_pDDSTwo, nil);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'CreateSurface FAILED');
      Exit;
    end;

  // Create and set the palette
  g_pDDPal := DDLoadPalette(g_pDD, szBitmap);
  if g_pDDPal = nil then
    begin
      Result := InitFail(h_Wnd, hRet, 'DDLoadPalette FAILED');
      Exit;
    end;
  hRet := IDirectDrawSurface4_SetPalette(g_pDDSPrimary, g_pDDPal);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetPalette FAILED');
      Exit;
    end;
  if not InitSurfaces then
    begin
      Result := InitFail(h_Wnd, hRet, 'InitSurfaces FAILED');
      Exit;
    end;

  // Create a timer to flip the pages
  if TIMER_ID <> SetTimer(h_Wnd, TIMER_ID, TIMER_RATE, nil) then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetTimer FAILED');
      Exit;
    end;

  Result := DD_OK;
end;

//-----------------------------------------------------------------------------
// Name: WinMain
// Desc: Initialization, message loop
//-----------------------------------------------------------------------------
var
  aMSG : MSG;
begin
  if InitApp(GetModuleHandle(nil), SW_SHOW) <> DD_OK then
    begin
      Exit;
    end;

  while GetMessage(@aMSG, 0, 0, 0) do
    begin
      TranslateMessage(aMSG);
      DispatchMessage(aMSG);
    end;

end.
