unit Serializer;

interface

uses
  SysUtils, Classes;

type
  ISerializer = interface;

  //   
  //     
  //        ,   
  ISerializable = interface
  ['{E523F5F0-13C0-4CB4-B1DD-BED0B701A177}']
  //        Context
    procedure Serialize(Context: ISerializer);
  //      (   )
    procedure Deserialize(Context: ISerializer);
  end;

  //  ,   ,
  //   ISerializable  ,
  //     
  ISerializer = interface
  ['{114653F7-DE4D-4954-B8AA-70A319F19258}']
  //  
    procedure SerializeObject(obj: ISerializable);
    procedure SerializeString(const s: string);
    procedure SerializeData(const Buffer; Size: integer);

  //  
    function DeserializeObject(TargetClass: TClass): TObject;
    function DeserializeString: string;
    procedure DeserializeData(var Buffer; Size: integer = 0);

  //       /
  //       Bitmap'
  //     Stream
  //   -  TStrings:   - ,
  //    
    function GetStream: TStream;
    property Stream: TStream read GetStream;
  end;

  ESerializationException = class(Exception)
  end;

  //      
  function NewSerializer(Stream: TStream; FinalizeStream: boolean = false): ISerializer;
  function NewDeserializer(Stream: TStream; FinalizeStream: boolean = false): ISerializer;

  //      
  procedure SerializeObject(Obj: ISerializable; Stream: TStream; FinalizeStream: boolean = false);
  function DeserializeObject(TargetClass: TClass; Stream: TStream; FinalizeStream: boolean = false): TObject;


implementation


const
  //      4   
  //    
  SERIALIZER_MAGIC_NUMBER = $44FFADB3;

type
  //   
  TSerializerToken = (
      stNone = 0,   //   
      stNil,        // nil   
      stObject,
      stString,
      stData
  );

{ ### TSerializerBase ######################################################## }

type
  TSerializerBase = class(TInterfacedObject, ISerializer)
  private
    fStream: TStream;
    fFinalizeStream: boolean;
    fObjectList: TStringList;   //   ,    
                                //    Dictionary<K,T>
                                //   ()  
  protected
    function FindObject(id: Integer; var obj: TObject): boolean;
    function FindId(obj: TObject; var id: integer): boolean;
    procedure AddObject(id: integer; obj: TObject);
    class procedure ProcessMagicNumber(Stream: TStream); virtual; abstract;
    property Stream: TStream read fStream;
  public
    constructor Create(aStream: TStream; aFinalizeStream: boolean); virtual;
    destructor Destroy; override;
    procedure SerializeObject(obj: ISerializable); virtual; abstract;
    procedure SerializeString(const s: string); virtual; abstract;
    procedure SerializeData(const Buffer; Size: integer); virtual; abstract;
    function DeserializeObject(TargetClass: TClass): TObject; virtual; abstract;
    function DeserializeString: string; virtual; abstract;
    procedure DeserializeData(var Buffer; Size: integer); virtual; abstract;
    function GetStream: TStream;
  end;

constructor TSerializerBase.Create(aStream: TStream; aFinalizeStream: boolean);
begin
  inherited Create;
  ProcessMagicNumber(aStream);
  fStream := aStream;
  fFinalizeStream := aFinalizeStream;
  fObjectList := TStringList.Create;
  fObjectList.Sorted := true;
  fObjectList.Duplicates := dupIgnore;
end;

destructor TSerializerBase.Destroy;
begin
  fObjectList.Free;
  if fFinalizeStream then
      fStream.Free;
  inherited;
end;

//    ID
function TSerializerBase.FindObject(id: Integer; var obj: TObject): boolean;
var index: integer;
begin
  if fObjectList.Find(IntToStr(id), index) then begin
      obj := fObjectList.Objects[index];
      Result := true
  end else begin
      Result := false
  end;
end;

//  ID  
function TSerializerBase.FindId(obj: TObject; var id: Integer): boolean;
var index: integer;
begin
  index := fObjectList.IndexOfObject(obj);
  if index >= 0 then begin
      id := StrToInt(fObjectList[index]);
      Result := true;
  end else begin
      Result := false;
  end;
end;

procedure TSerializerBase.AddObject(id: Integer; obj: TObject);
begin
  fObjectList.AddObject(IntToStr(id), obj);
end;

function TSerializerBase.GetStream: TStream;
begin
  Result := fStream;
end;


{ ### TSerializer ############################################################ }

type
  TSerializer = class(TSerializerBase)
  private
    fIdCounter: integer;
  protected
    function NextId: integer;
    class procedure ProcessMagicNumber(Stream: TStream); override;
  public
    procedure SerializeObject(obj: ISerializable); override;
    procedure SerializeString(const s: string); override;
    procedure SerializeData(const Buffer; Size: integer); override;
    function DeserializeObject(TargetClass: TClass): TObject; override;
    function DeserializeString: string; override;
    procedure DeserializeData(var Buffer; Size: integer); override;
  end;

function TSerializer.NextId: integer;
begin
  inc(fIdCounter);
  Result := fidCounter;
end;

class procedure TSerializer.ProcessMagicNumber(Stream: TStream);
var MagicNumber: cardinal;
begin
  MagicNumber := SERIALIZER_MAGIC_NUMBER;
  Stream.Write(MagicNumber, sizeof(MagicNumber));
end;

procedure TSerializer.SerializeObject(obj: ISerializable);
var id: Integer;
    token: TSerializerToken;
begin
  if obj = nil then begin
      token := stNil;
      Stream.Write(token, sizeof(token));
  end else begin
      token := stObject;
      Stream.Write(token, sizeof(token));
      if FindId(TObject(obj), id) then begin
          Stream.Write(id, sizeof(id));
      end else begin
          id := NextId;
          Stream.Write(id, sizeof(id));

          AddObject(id, TObject(obj));
          obj.Serialize(Self);
      end;
  end;
end;

procedure TSerializer.SerializeString(const s: string);
var token: TSerializerToken;
    len: integer;
begin
  token := stString;
  Stream.Write(token, sizeof(token));
  len := Length(s);
  Stream.Write(len, sizeof(len));
  Stream.Write(pointer(s)^, len * sizeof(char));
end;

procedure TSerializer.SerializeData(const Buffer; Size: Integer);
var token: TSerializerToken;
begin
  token := stData;
  Stream.Write(token, sizeof(token));
  Stream.Write(Size, sizeof(Size));
  Stream.Write(Buffer, Size);
end;

function TSerializer.DeserializeObject(TargetClass: TClass): TObject;
begin
  raise ESerializationException.Create('Cannot deserialize object: stream was opend with serialization reason.');
end;

function TSerializer.DeserializeString: string;
begin
  raise ESerializationException.Create('Cannot deserialize string: stream was opend with serialization reason.');
end;

procedure TSerializer.DeserializeData(var Buffer; Size: Integer);
begin
  raise ESerializationException.Create('Cannot deserialize data: stream was opend with serialization reason.');
end;

{ ### TDeserializer ########################################################## }

type
  TDeserializer = class(TSerializerBase)
  protected
    class procedure ProcessMagicNumber(Stream: TStream); override;
    function NextToken: TSerializerToken;
  public
    procedure SerializeObject(obj: ISerializable); override;
    procedure SerializeString(const s: string); override;
    procedure SerializeData(const Buffer; Size: integer); override;
    function DeserializeObject(TargetClass: TClass): TObject; override;
    function DeserializeString: string; override;
    procedure DeserializeData(var Buffer; Size: integer); override;
  end;

class procedure TDeserializer.ProcessMagicNumber(Stream: TStream);
var MagicNumber: cardinal;
begin
  Stream.Read(MagicNumber, sizeof(MagicNumber));
  if MagicNumber <> SERIALIZER_MAGIC_NUMBER then
      raise ESerializationException.Create('Invalid input stream.');  
end;

function TDeserializer.NextToken: TSerializerToken;
begin
  fStream.Read(Result, sizeof(Result));
end;

procedure TDeserializer.SerializeObject(obj: ISerializable);
begin
  raise ESerializationException.Create('Cannot serialize object: stream was opend with deserialization reason.');
end;

procedure TDeserializer.SerializeString(const s: string);
begin
  raise ESerializationException.Create('Cannot serialize string: stream was opend with deserialization reason.');
end;

procedure TDeserializer.SerializeData(const Buffer; Size: Integer);
begin
  raise ESerializationException.Create('Cannot serialize data: stream was opend with deserialization reason.');
end;

function TDeserializer.DeserializeObject(TargetClass: TClass): TObject;
var id: integer;
    obj: TObject;
    iobj: ISerializable;
begin
  case NextToken of
      stNil: begin
          Result := nil;
      end;
      stObject: begin
          Stream.Read(id, sizeof(id));
          if FindObject(id, obj) then begin
              Result := obj
          end else begin
              obj := TargetClass.NewInstance;
              if obj.GetInterface(ISerializable, iobj) then begin
                  AddObject(id, obj);
                  try
                      iobj.Deserialize(Self);
                  except
                      obj.FreeInstance;
                      raise;
                  end;
              end else begin
                  obj.FreeInstance;
                  raise ESerializationException.CreateFmt('Cannot deserialize object: specified class "%s" doesnot support ISerializeble.', [TargetClass.ClassName]);
              end;
              Result := obj;
          end;
      end;
      else begin
          raise ESerializationException.Create('Cannot deserialize object: object token expected.');
      end;
  end;
end;

function TDeserializer.DeserializeString: string;
var len: integer;
begin
  if NextToken = stString then begin
      Stream.Read(len, sizeof(len));
      SetLength(Result, len);
      Stream.Read(pointer(Result)^, len * sizeof(char));
  end else begin
      raise ESerializationException.Create('Cannot deserialize string: string token expected.');
  end;
end;

procedure TDeserializer.DeserializeData(var Buffer; Size: integer);
var dataSize: integer;
begin
  if NextToken = stData then begin
      Stream.Read(dataSize, sizeof(dataSize));
      if Size <> 0 then begin
          if Size <> dataSize then 
              raise ESerializationException.Create('Cannot deserialize data block: unexpected data block size.');
      end;
      Stream.Read(Buffer, dataSize);
  end else begin
      raise ESerializationException.Create('Cannot deserialize data block: data token expected.');
  end;
end;

{ ############################################################################ }

function NewSerializer(Stream: TStream; FinalizeStream: boolean = false): ISerializer;
begin
  Result := TSerializer.Create(Stream, FinalizeStream);
end;

function NewDeserializer(Stream: TStream; FinalizeStream: boolean = false): ISerializer;
begin
  Result := TDeserializer.Create(Stream, FinalizeStream);
end;

procedure SerializeObject(Obj: ISerializable; Stream: TStream; FinalizeStream: boolean = false);
var Serializer: ISerializer;
begin
  Serializer := TSerializer.Create(Stream, FinalizeStream);
  Serializer.SerializeObject(Obj);
end;

function DeserializeObject(TargetClass: TClass; Stream: TStream; FinalizeStream: boolean = false): TObject;
var Deserializer: ISerializer;
begin
  Deserializer := TDeserializer.Create(Stream, FinalizeStream);
  Result := Deserializer.DeserializeObject(TargetClass);
end;

end.
