unit Pipes;

interface

uses
    Windows, SysUtils, Classes;

type
    TPipeCommunication = class;

    TStringRecievedEvent = procedure (sender: TPipeCommunication; const s: string) of object;

    TPipeCommunication = class
    private
        fPipeName: string;
        fIsServer: boolean;
        fHandle: integer;
        fServerThread: TThread;
        fOnStringRecieved: TStringRecievedEvent;
        function ClientConnected: boolean;
        procedure ClientDisconnected;
        function Read(var s: string): boolean;
        function Write(const s: string): boolean;
        procedure DoStringRecieved(const s: string);
    public
        constructor Create(const PipeName: string);
        destructor Destroy; override;
        procedure SendString(const s: string); virtual;
        procedure Flush;
        property PipeName: string read fPipeName;
        property IsServer: boolean read fIsServer;
        property Handle: integer read fHandle;
        property OnStringRecieved: TStringRecievedEvent read fOnStringRecieved write fOnStringRecieved;
    end;

implementation

const
    PipeBufferSize = 4096;    //  
    PipeTimeout = 4000;

//   -  
function CreatePipe(const PipeName: string): integer;
var handle: THandle;
    QualifiedPipeName: PChar;
begin
    QualifiedPipeName := PChar('\\.\pipe\' + PipeName);
    handle := Windows.CreateNamedPipe(
        QualifiedPipeName,
        PIPE_ACCESS_DUPLEX,
        PIPE_TYPE_BYTE or
        PIPE_READMODE_BYTE or
        PIPE_WAIT,
        1,
        PipeBufferSize,
        PipeBufferSize,
        PipeTimeout,
        nil);
    if handle = INVALID_HANDLE_VALUE then
        Result := 0
    else
        Result := integer(handle);
end;

//   -  
function OpenPipe(const PipeName: string): integer;
var handle: integer;
    QualifiedPipeName: string;
begin
    QualifiedPipeName := '\\.\pipe\' + PipeName;
    handle := SysUtils.FileOpen(QualifiedPipeName, fmOpenWrite);
    if handle = -1 then
        Result := 0
    else
        Result := handle;
end;

//    -  
function WaitForClient(handle: integer): boolean;
begin
    Result := Windows.ConnectNamedPipe(handle, nil);
end;

//   -  
procedure DisconnectClient(handle: integer);
begin
    Windows.DisconnectNamedPipe(handle);
end;

//    
function WriteString(handle: integer; const s: string): boolean;
var writeResult: integer;
    LineLength: integer;
begin
    LineLength := Length(s);
    if (0 < LineLength) and (LineLength < PipeBufferSize) then begin
    //   
        SysUtils.FileWrite(handle, LineLength, sizeof(LineLength));
    //   
        writeResult := SysUtils.FileWrite(handle, pointer(s)^, LineLength * sizeof(Char));
        Result := writeResult <> -1;
    end else
        Result := false;
end;

//    
function ReadString(handle: integer; var s: string): boolean;
var readResult: integer;
    LineLength: integer;
begin
    Result := false;
    //  
    readResult := SysUtils.FileRead(handle, LineLength, sizeof(LineLength));
    if readResult = -1 then
        Exit;
    if LineLength >= PipeBufferSize then
        Exit;
    SetLength(s, LineLength);
    //  
    readResult := SysUtils.FileRead(handle, pointer(s)^, LineLength * sizeof(Char));
    Result := readResult <> -1;
end;

//   
procedure FlushPipeBuffers(handle: integer);
begin
    Windows.FlushFileBuffers(handle);
end;

//  
procedure ClosePipe(handle: integer);
begin
    Windows.CloseHandle(handle);
end;

//       
type
    TPipeCommunicationServerThread = class(TThread)
    private
        fOwner: TPipeCommunication;
        fRecievedString: string;
        procedure StringRecieved;
    protected
        procedure Execute; override;
    public
        constructor Create(Owner: TPipeCommunication);
        property Owner: TPipeCommunication read fOwner;
    end;

constructor TPipeCommunicationServerThread.Create(Owner: TPipeCommunication);
begin
    inherited Create(false);
    fOwner := Owner;
end;

procedure TPipeCommunicationServerThread.StringRecieved;
begin
    Owner.DoStringRecieved(fRecievedString);
end;

procedure TPipeCommunicationServerThread.Execute;
begin
    while not Terminated do begin
         //   
         if Owner.ClientConnected() then begin
             //    
             while Owner.Read(fRecievedString) do
                 Synchronize(StringRecieved);  //     
             //    
             Owner.ClientDisconnected;
         end else begin
             Sleep(PipeTimeout div 2);
         end
    end
end;

constructor TPipeCommunication.Create(const PipeName: string);
var handle: integer;
begin
    //    
    handle := OpenPipe(PipeName);
    if handle = 0 then begin
        //    -  
        handle := CreatePipe(PipeName);
        //    
        fIsServer := true;
    end;
    fPipeName := PipeName;
    fHandle := handle;
    if fIsServer and (fHandle <> 0) then
        //    
        fServerThread := TPipeCommunicationServerThread.Create(Self);
end;

destructor TPipeCommunication.Destroy;
var ExitCode: cardinal;
begin
    if Assigned(fServerThread) then begin
        //   
        ExitCode := 0;
        Windows.GetExitCodeThread(fServerThread.Handle, ExitCode);
        Windows.TerminateThread(fServerThread.Handle, ExitCode);
        fServerThread.Free;
    end;
    //   ( )
    ClosePipe(fHandle);
    fHandle := 0;
end;

function TPipeCommunication.ClientConnected(): boolean;
begin
    Result := WaitForClient(fHandle);
end;

procedure TPipeCommunication.ClientDisconnected;
begin
    DisconnectClient(fHandle);
end;

function TPipeCommunication.Read(var s: string): boolean;
begin
    Result := ReadString(fHandle, s);
end;

function TPipeCommunication.Write(const s: string): boolean;
begin
    Result := WriteString(fHandle, s);
end;

procedure TPipeCommunication.DoStringRecieved(const s: string);
begin
    if Assigned(fOnStringRecieved) then
        fOnStringRecieved(Self, s);
end;

procedure TPipeCommunication.SendString(const s: string);
begin
    Write(s);
end;

procedure TPipeCommunication.Flush;
begin
    FlushPipeBuffers(fHandle);
end;

end.
