Unit Equation;

Interface
Type
  Float = Double;

Function solveQuadratic(Var vx, vy): Byte;
Function solveCubic(Var vx, vy): Byte;

Function solveQuarticAlgebra(Var vx, vres): Byte;
Function solveQuarticVieta(Var vx, vres): Byte;

Implementation
Type
  arrFloat =
    Array[ 0 .. Pred(2 * maxInt Div SizeOf(Float))] Of Float;

Const
  Epsilon     = 1.0E-10;
  maxDistance = 1.0E+07;


{
  Solve the quadratic equation:

  x[0] * x^2 + x[1] * x + x[2] = 0.

  The value returned by this function is the number of real roots.
  The roots themselves are returned in y[0], y[1].
}
Function solveQuadratic(Var vx, vy): Byte;
  Var
    x: arrFloat Absolute vx;
    y: arrFloat Absolute vy;

  Var
    a, b, c, Desc: Float;
    T: Float;
  Begin
    a := x[0]; b := -x[1]; c := x[2];
    If a = 0 Then
      Begin
        solveQuadratic := 0;
        If b = 0.0 Then Exit;

        solveQuadratic := 1;
        y[0] := c / b; Exit
      End;

    solveQuadratic := 0;
    Desc := Sqr(b) - 4 * a * c;
    If Desc < 0.0 Then Exit
    Else
      If Abs(Desc) < Epsilon Then
        Begin
          solveQuadratic := 1;
          y[0] := 0.5 * b / a;
          Exit
        End;

    Desc := Sqrt(Desc);
    T := 2 * a;
    y[0] := (b + Desc) / T;
    y[1] := (b - Desc) / T;
    solveQuadratic := 2
  End;


{
  Solve the cubic equation:

  x[0] * x^3 + x[1] * x^2 + x[2] * x + x[3] = 0.

  The result of this function is an integer that tells how many real
  roots exist.  Determination of how many are distinct is up to the
  process that calls this routine.  The roots that exist are stored
  in (y[0], y[1], y[2]).

  Note: this function relies very heavily on trigonometric functions and
  the square root function. If an alternative solution is found that does
  not rely on transcendentals this code should be replaced.
}
Function solveCubic(Var vx, vy): Byte;
  Var
    x: arrFloat Absolute vx;
    y: arrFloat Absolute vy;

  Function ArcCos(x: Float): Float;
    Begin
      If x = 0 Then
        ArcCos := Pi / 2
      Else
        ArcCos := ArcTan( Sqrt(1 - Sqr(x)) / x ) + Pi * Byte(x < 0)
    End;

  Function Pow(a, x: Float): Float;
    Begin
      Pow := Exp( x * Ln(a) )
    End;

  Var
    a0, a1, a2, a3, an: Float;
    sq, q, q3, r, r2, sqrA: Float;
    theta, Desc: Float;
  Begin
    a0 := x[0];
    If a0 = 0.0 Then
      Begin
        solveCubic := solveQuadratic(x[1], y);
        Exit
      End
    Else
      Begin
        a1 := x[1] / a0;
        a2 := x[2] / a0;
        a3 := x[3] / a0;
      End;

    sqrA := sqr(a1);
    q := (sqrA - 3.0 * a2) / 9.0;
    r := (2.0 * sqrA * a1 - 9.0 * a1 * a2 + 27.0 * a3) / 54.0;
    q3 := sqr(q) * q;
    r2 := sqr(r);
    Desc := q3 - r2;
    an := a1 / 3.0;
    If Desc >= 0 Then
      Begin
        { Three real roots }
        solveCubic := 3;

        Desc := r / sqrt(q3);
        theta := ArcCos(Desc) / 3;
        sq := -2.0 * Sqrt(q);
        y[0] := sq * Cos(theta) - an;
        y[1] := sq * Cos(theta + (2 * Pi / 3)) - an;
        y[2] := sq * Cos(theta + (4 * Pi / 3)) - an;
        Exit
      End
    Else
      Begin
        solveCubic := 1;

        sq := Pow(Sqrt(r2 - q3) + Abs(r), 1 / 3);
        If r < 0 Then
          y[0] := (sq + q / sq) - an
        Else
          y[0] := -(sq + q / sq) - an;
        Exit
      End
  End;




(*
  Additional functions for solveQuartic.
*)
Const
  fudgeFactorFirst  =  1.0E+12;
  fudgeFactorSecond = -1.0E-05;
  fudgeFactorThird  =  1.0E-07;


Function DifficultCoeffs(n: Integer; Var vx): Boolean;
  Var
    x: arrFloat Absolute vx;

  Var
    i: Integer;
    biggest: Float;
  Begin
    biggest := Abs(x[0]);
    For i := 1 To n Do
      If Abs(x[i]) > biggest Then
        biggest := Abs(x[i]);

    DifficultCoeffs := False;
    If biggest = 0.0 Then Exit;

    For i := 0 To n Do
      If x[i] <> 0.0 Then
        If Abs(biggest / x[i]) > fudgeFactorFirst Then
          Begin
            DifficultCoeffs := True;
            x[i] := 0.0
          End
  End;


{
  Root solver based on the STURM sequences for a polynomial
}
Function PolySolve(order: Integer;
         Var vCoeffs, vroots): Integer;
  Var
    Coeffs: arrFloat Absolute vCoeffs;
    roots: arrFloat Absolute vroots;

  Const
    maxOrder = 15;
    maxIterations = 50;

  Type
    TPolynomial =
      Record
        Ordr: Integer;
        coef: Array[ 0 .. maxOrder ] Of Float;
      End;
  Var
    sseq: Array[ 0 .. maxOrder ] Of TPolynomial;


    Function modp(iu, iv, ir: Integer): Integer;
      Var
        i, j, k: Integer;
      Begin

        For i := 0 To Pred(sseq[iu].Ordr) Do
          move( sseq[iu + i], sseq[ir + i],
                SizeOf(TPolynomial) );

        If sseq[iv].coef[ sseq[iv].Ordr ] < 0 Then
          Begin
            k := sseq[iu].Ordr - sseq[iv].Ordr - 1;
            While k >= 0 Do
              Begin
                sseq[ir].coef[k] := - sseq[ir].coef[k];
                Dec(k, 2)
              End;

            For k := sseq[iu].Ordr - sseq[iv].Ordr DownTo 0 Do
              For j := sseq[iv].Ordr + k - 1 DownTo k Do
                sseq[ir].coef[j] :=
                  - sseq[ir].coef[j]
                  - sseq[ir].coef[ sseq[iv].Ordr + k ] *
                    sseq[iv].coef[j - k];
          End;

        If sseq[iv].coef[ sseq[iv].Ordr ] < 0 Then
          Begin
            k := sseq[iu].Ordr - sseq[iv].Ordr - 1;
            While k >= 0 Do
              Begin
                sseq[ir].coef[k] := - sseq[ir].coef[k];
                Dec(k, 2)
              End;

            For k := sseq[iu].Ordr - sseq[iv].Ordr DownTo 0 Do
              For j := sseq[iv].Ordr + k - 1 DownTo k Do
                sseq[ir].coef[j] :=
                  - sseq[ir].coef[j]
                  - sseq[ir].coef[ sseq[iv].Ordr + k ] *
                    sseq[iv].coef[j - k];

          End
        Else
          Begin
            For k := sseq[iu].Ordr - sseq[iv].Ordr DownTo k Do
              For j := sseq[iv].Ordr + k - 1 DownTo k Do
                sseq[ir].coef[j] :=
                  sseq[ir].coef[j] - sseq[ir].coef[ sseq[iv].Ordr + k ] *
                                     sseq[iv].coef[j - k];
          End;

        k := sseq[iv].Ordr - 1;
        While (k > 0) and (Abs(sseq[ir].coef[k]) < Epsilon) Do
          Begin
            sseq[ir].coef[k] := 0;
            Dec(k)
          End;

        If k < 0 Then
          sseq[ir].Ordr := 0
        Else sseq[ir].Ordr := k;

        modp := sseq[ir].Ordr
      End;


    { build the sturmian sequence for a polynomial }
    Function buildsturm(iOrd: Integer): Integer;
      Var
        f: Float;
        i, ifp, ifc, isp: Integer;
      Begin
        sseq[0].Ordr := iOrd;
        sseq[1].Ordr := iOrd - 1;

        f := Abs(sseq[0].coef[iOrd] * iOrd);

        ifp := 0; ifc := 1;
        For i := 1 To iOrd Do
          Begin
            sseq[1].coef[ifp] :=
              sseq[0].coef[ifc] * (i / f);
            Inc(ifp); Inc(ifc)
          End;

        isp := 2;
        While modp( isp - 2, isp - 1, isp ) > 0 Do
          Begin
            f := - Abs( sseq[isp].coef[ sseq[isp].Ordr ] );
            For ifp := sseq[isp].Ordr DownTo 0 Do
              sseq[isp].coef[ifp] := sseq[isp].coef[ifp] / f
          End;

        sseq[isp].coef[0] := - sseq[isp].coef[0];
        buildsturm := isp
      End;

    { Find out how many visible intersections there are }
    Function VisibleRoots(np: Integer;
             Var atzer, atpos: Integer): Integer;
      Var
        s, atposinf, atzero: Integer;
        f, lf: Float;
      Begin
        atposinf := 0; atzero := 0;

        { Changes at positve infinity }
        lf := sseq[0].coef[ sseq[0].Ordr ];
        For s := 1 To np Do
          Begin
            f := sseq[s].coef[ sseq[s].Ordr ];
            If (lf = 0.0) or (lf * f < 0)
              Then Inc(atposinf);
            lf := f
          End;

        { Changes at zero }
        lf := sseq[0].coef[0];
        For s := 1 To np Do
          Begin
            f := sseq[s].coef[0];
            If (lf = 0) or (lf * f < 0)
              Then Inc(atzero);
            lf := f
          End;

        atzer := atzero;
        atpos := atposinf;
        VisibleRoots := atzero - atposinf
      End;

    Function PolyEval(x: Float; n: Integer;
             Var vCoeffs): Float;
      Var
        Coeffs: arrFloat Absolute vCoeffs;

      Var
        val: Float;
        i: Integer;
      Begin
        val := Coeffs[n];
        For i := n - 1 DownTo 0 Do
          val := val * x + Coeffs[i];
        PolyEval := val
      End;


    { Returns the number of sign changes in the
      STURM sequence in sseq at the value a }
    Function numchanges(np: Integer; a: Float): Integer;
      Var
        s, changes: Integer;
        f, lf: Float;
      Begin
        changes := 0;
        lf := PolyEval(a, sseq[0].Ordr, sseq[0].coef);

        For s := 1 To np Do
          Begin
            f := PolyEval(a, sseq[s].Ordr, sseq[s].coef);
            If (lf = 0.0) or (lf * f < 0)
              Then Inc(changes);
            lf := f
          End;

        numchanges := changes;
      End;



    { Uses a bisection based on the sturm sequence for the polynomial }
    Function sBisect(np: Integer; min_value, max_value: Float;
             atmin, atmax: Integer; Var vRoots): Integer;
      Var
        roots: arrFloat Absolute vRoots;

        { Close in on a root by using regula-falsa }
        Function RegulaFalsa(Order: Integer; Var vCoef;
                 a, b: Float; Var val: Float ): Boolean;
          Var
            Coef: arrFloat Absolute vCoef;

          Var
            its: Integer;
            fa, fb, x, fx, lfx: Float;
          Begin
            fa := PolyEval(a, order, Coef);
            fb := PolyEval(b, order, Coef);
            RegulaFalsa := False;
            If fa * fb > 0 Then Exit;

            RegulaFalsa := True;
            If Abs(fa) < Epsilon Then
              Begin
                val := a; Exit
              End;

            If Abs(fb) < Epsilon Then
              Begin
                val := b; Exit
              End;

            lfx := fa;
            For its := 0 To maxIterations Do
              Begin
                x := (fb * a - fa * b) / (fb - fa);
                fx := PolyEval(x, order, coef);

                If Abs(x) > Epsilon Then
                  Begin
                    If Abs(fx / x) < Epsilon Then
                      Begin
                        val := x; Exit
                      End;
                  End
                Else
                  If Abs(fx) < Epsilon Then
                    Begin
                      val := x; Exit
                    End;

                If fa < 0 Then
                  If fx < 0 Then
                    Begin
                      a := x; fa := fx;
                      If (lfx * fx) > 0
                        Then fb := fb / 2
                    End
                  Else
                    Begin
                      b := x; fb := fx;
                      If (lfx * fx) > 0
                        Then fa := fa / 2
                    End
                Else
                  If fx < 0 Then
                    Begin
                      b := x; fb := fx;
                      If (lfx * fx) > 0
                        Then fa := fa / 2
                    End
                  Else
                    Begin
                      a := x; fa := fx;
                      If (lfx * fx) > 0
                        Then fb := fb / 2
                    End;

                If Abs(b-a) < Epsilon Then
                  Begin
                    val := x; Exit
                  End;

                lfx := fx
              End;

            RegulaFalsa := False
          End;


      Var
        n1, n2, atmid, its: Integer;
        mid: Float;
      Begin
        If atmin - atmax = 1 Then
          Begin
            sBisect := 1;

            If RegulaFalsa(sseq[0].Ordr, sseq[0].coef,
                           min_value, max_value, roots[0]) Then Exit
            Else
              Begin
                { regula-falsa failed, so now find it by bisection }
                For its := 0 To Pred(maxIterations) Do
                  Begin
                    mid := (min_value + max_value) / 2;
                    atmid := numchanges(np, mid);
                    If Abs(mid) > Epsilon Then
                      Begin
                        If Abs((max_value - min_value) / mid) < Epsilon Then
                          Begin
                            roots[0] := mid; Exit
                          End
                      End
                    Else
                      If Abs(max_value - min_value) < Epsilon Then
                        Begin
                          roots[0] := mid; Exit
                        End
                      Else
                        If atmin - atmid = 0 Then
                          min_value := mid
                        Else max_value := mid
                  End;

                { bisection took too long }
                roots[0] := mid; Exit
              End
          End;

        { There is more than one root in the interval.
          Bisect to find new intervals }
        For its := 0 To Pred(maxIterations) Do
          Begin
            mid := (min_value + max_value) / 2;
            atmid := numchanges(np, mid);
            n1 := atmin - atmid;
            n2 := atmid - atmax;
            If (n1 <> 0) and (n2 <> 0) Then
              Begin
                n1 := sBisect(np, min_value, mid, atmin, atmid, roots);
                n2 := sBisect(np, mid, max_value, atmid, atmax, roots[n1]);
                sBisect := n1 + n2; Exit
              End;

            If n1 = 0 Then
              min_value := mid
            Else
              max_value := mid
          End;

        { bisection took too long }
        roots[0] := mid;
        sBisect := 1
      End;


  Var
    i, np, nroots: Integer;
    atmin, atmax: Integer;
    min_value, max_value: Float;
  Begin
    For i := 0 To order Do
      sseq[0].coef[ order - i ] := Coeffs[i];

    np := buildsturm(order);

    { total number of visible roots }
    nroots := VisibleRoots(np, atmin, atmax);

    PolySolve := 0;
    If nroots = 0 Then Exit;

    { bracket the roots }
    min_value := 0.0;
    max_value := maxDistance;

    atmin := numchanges(np, min_value);
    atmax := numchanges(np, max_value);
    nroots := atmin - atmax;
    If nroots = 0 Then Exit;

    { perform the bisection }
    PolySolve :=
      sbisect(np, min_value, max_value, atmin, atmax, roots);
  End;


{
  solving quartics algebraically (method of Lodovico Ferrari)
}
Function solveQuarticAlgebra(Var vx, vres): Byte;
  Var
    x: arrFloat Absolute vx;
    results: arrFloat Absolute vres;

  Var
    i: Integer;
    a0, a1, y, d1, x1, t1, t2: Float;
    c0, c1, c2, c3, c4, d2, q1, q2: Float;
    cubic: Array[0 .. 3] Of Float;
    roots: Array[0 .. 2] Of Float;
  Begin
    If DifficultCoeffs(4, x) Then
      Begin
        If Abs(x[0]) < Epsilon Then
          If Abs(x[1]) < Epsilon Then
            Begin
              solveQuarticAlgebra := solveQuadratic(x[2], results); Exit
            End
          Else
            Begin
              solveQuarticAlgebra := solveCubic(x[1], results); Exit
            End

        Else
          Begin
            solveQuarticAlgebra := PolySolve(4, x, results); Exit
          End
      End;

    c0 := x[0];
    If Abs(c0) < Epsilon Then
      Begin
        solveQuarticAlgebra := solveCubic(x[1], results); Exit
      End
    Else
      If Abs(x[4]) < Epsilon Then
        Begin
          solveQuarticAlgebra := solveCubic(x, results); Exit
        End
      Else
        Begin
          c1 := x[1] / c0;
          c2 := x[2] / c0;
          c3 := x[3] / c0;
          c4 := x[4] / c0;
        End;

      (*
        The first step is to take the original equation:

	   x^4 + b*x^3 + c*x^2 + d*x + e = 0

        and rewrite it as:

	   x^4 + b*x^3 = -c*x^2 - d*x - e,

        adding (b*x/2)^2 + (x^2 + b*x/2)y + y^2/4 to each side gives a
        perfect square on the lhs:

	   (x^2 + b*x/2 + y/2)^2 = (b^2/4 - c + y)x^2 + (b*y/2 - d)x + y^2/4 - e

        By choosing the appropriate value for y, the rhs can be made a perfect
        square also.  This value is found when the rhs is treated as a quadratic
        in x with the discriminant equal to 0.  This will be true when:

	   (b*y/2 - d)^2 - 4.0 * (b^2/4 - c*y)*(y^2/4 - e) = 0, or

	   y^3 - c*y^2 + (b*d - 4*e)*y - b^2*e + 4*c*e - d^2 = 0.

        This is called the resolvent of the quartic equation
      *)

      a0 := 4.0 * c4;
      cubic[0] := 1.0;
      cubic[1] := -1.0 * c2;
      cubic[2] := c1 * c3 - a0;
      cubic[3] := a0 * c2 - Sqr(c1) * c4 - Sqr(c3);
      i := solveCubic(cubic[0], roots[0]);

      solveQuarticAlgebra := 0;
      If i > 0 Then
        y := roots[0]
      Else
        Exit;

      (*
        What we are left with is a quadratic squared on the lhs and a
        linear term on the right.  The linear term has one of two signs,
        take each and add it to the lhs.  The form of the quartic is now:

	   a' = b^2/4 - c + y,    b' = b*y/2 - d, (from rhs quadritic above)

	   (x^2 + b*x/2 + y/2) = +sqrt(a'*(x + 1/2 * b'/a')^2), and
	   (x^2 + b*x/2 + y/2) = -sqrt(a'*(x + 1/2 * b'/a')^2).

        By taking the linear term from each of the right hand sides and
        adding to the appropriate part of the left hand side, two quadratic
        formulas are created.  By solving each of these the four roots of
        the quartic are determined
      *)

      i := 0;
      a0 := c1 / 2.0;
      a1 := y / 2.0;

      t1 := Sqr(a0) - c2 + y;
      If t1 < 0.0 Then
        Begin
          If t1 > fudgeFactorSecond
            Then t1 := 0.0
          Else
            { First Special case, a' < 0 means all roots are complex }
            Exit
        End;

      If t1 < fudgeFactorThird Then
        Begin
          (*
            Second special case, the "x" term on the right hand side above
	    has vanished.  In this case:
		      (x^2 + b*x/2 + y/2) = +sqrt(y^2/4 - e), and
		      (x^2 + b*x/2 + y/2) = -sqrt(y^2/4 - e).
          *)
          t2 := Sqr(a1) - c4;
          If t2 < 0.0 Then Exit;

          x1 := 0.0;
          d1 := Sqrt(t2)
        End
      Else
        Begin
          x1 := Sqrt(t1);
          d1 := 0.5 * (a0 * y - c3) / x1
        End;

      { Solve the first quadratic }
      q1 := -a0 - x1;
      q2 := a1 + d1;
      d2 := Sqr(q1) - 4.0 * q2;

      If d2 >= 0.0 Then
        Begin
          d2 := Sqrt(d2);
          results[0] := 0.5 * (q1 + d2);
          results[1] := 0.5 * (q1 - d2);
          i := 2
        End;

      { Solve the second quadratic }
      q1 := q1 + x1 + x1;
      q2 := a1 - d1;
      d2 := Sqr(q1) - 4.0 * q2;
      If d2 >= 0.0 Then
        Begin
          d2 := Sqrt(d2);
          results[i] := 0.5 * (q1 + d2); Inc(i);
          results[i] := 0.5 * (q1 - d2); Inc(i)
        End;

    solveQuarticAlgebra := i
  End;


{
  Solve a quartic using the method of Francois Vieta (Circa 1735)
}
Function solveQuarticVieta(Var vx, vres): Byte;
  Var
    x: arrFloat Absolute vx;
    results: arrFloat Absolute vres;

  Var
    i: Integer;
    c0, c1, c2, c3, c4: Float;
    c12, z, p, q, q1, q2, r, d1, d2: Float;

    cubic: Array[0 .. 3] Of Float;
    roots: Array[0 .. 2] Of Float;
  Begin
    { Figure out the size difference between coefficients }
    If DifficultCoeffs(4, x) Then

      If Abs(x[0]) < Epsilon Then
        If Abs(x[1]) < Epsilon Then
          Begin
            solveQuarticVieta := solveQuadratic(x[2], results);
            Exit
          End
        Else
          Begin
            solveQuarticVieta := solveCubic(x[1], results);
            Exit;
          End
      Else
        Begin
          solveQuarticVieta := PolySolve(4, x, results);
          Exit
        End;

    { See if the high order term has vanished }
    c0 := x[0];
    If Abs(c0) < Epsilon Then
      Begin
        solveQuarticVieta := solveCubic(x[1], results); Exit
      End;

    { See if the constant term has vanished }
    If Abs(x[4]) < Epsilon Then
      Begin
        solveQuarticVieta := solveCubic(x, results); Exit
      End;

    { Make sure the quartic has a leading coefficient of 1.0 }
    c1 := x[1] / c0;
    c2 := x[2] / c0;
    c3 := x[3] / c0;
    c4 := x[4] / c0;

    { Compute the cubic resolvant }
    c12 := c1 * c1;
    p := -0.375 * c12 + c2;
    q := 0.125 * c12 * c1 - 0.5 * c1 * c2 + c3;
    r := -0.01171875 * c12 * c12 + 0.0625 * c12 * c2 - 0.25 * c1 * c3 + c4;

    cubic[0] := 1.0;
    cubic[1] := -0.5 * p;
    cubic[2] := -r;
    cubic[3] := 0.5 * r * p - 0.125 * Sqr(q);
    i := solveCubic(cubic, roots);

    solveQuarticVieta := 0;
    If i > 0 Then
      z := roots[0]
    Else Exit;

    d1 := 2.0 * z - p;

    If d1 < 0.0 Then
      If d1 > - Epsilon Then
        d1 := 0.0
      Else Exit;

    If d1 < Epsilon Then
      Begin
        d2 := Sqr(z) - r;
        If d2 < 0.0 Then Exit;
        d2 := Sqrt(d2)
      End
    Else
      Begin
        d1 := Sqrt(d1);
        d2 := 0.5 * q / d1
      End;

    { Set up useful values for the quadratic factors }
    q1 := Sqr(d1);
    q2 := -0.25 * c1;
    i := 0;

    { Solve the first quadratic }
    p := q1 - 4.0 * (z - d2);
    If p = 0 Then
      Begin
        results[i] := -0.5 * d1 - q2; Inc(i)
      End
    Else
      If p > 0 Then
        Begin
          p := Sqrt(p);
          results[i] := -0.5 * (d1 + p) + q2; Inc(i);
          results[i] := -0.5 * (d1 - p) + q2; Inc(i);
        End;

    { Solve the second quadratic }
    p := q1 - 4.0 * (z + d2);
    If p = 0 Then
      Begin
        results[i] := 0.5 * d1 - q2; Inc(i)
      End
    Else
      If p > 0 Then
        Begin
          p := Sqrt(p);
          results[i] := 0.5 * (d1 + p) + q2; Inc(i);
          results[i] := 0.5 * (d1 - p) + q2; Inc(i)
        End;

    solveQuarticVieta := i
  End;

End.
