with Ada.Finalization;
with Ada.Strings.UTF_Encoding;
use Ada.Strings.UTF_Encoding;
with Ada.Strings.UTF_Encoding.Wide_Wide_Strings;
use Ada.Strings.UTF_Encoding.Wide_Wide_Strings;
with Ada.Characters.Wide_Latin_1;
with Ada.Wide_Wide_Text_IO;

with System; use System;

with Win32.Wincon;
with Win32.Winbase;
with Win32.Winnt;
with Win32.Winerror;

with Windows_Support; use Windows_Support;

package body Windows_Support.Console is

   use Ada;
   use Win32;
   use type Win32.DWORD;

   NUL_16 : Wide_Character renames Ada.Characters.Wide_Latin_1.NUL;

   -----------------------
   -- Set_Console_Title --
   -----------------------

   procedure Set_Console_Title (New_Title : Wide_Wide_String) is
      New_Title_16Z : aliased constant UTF_16_Wide_String :=
        Encode (New_Title) & NUL_16;
   begin
      Check_OS_Error
        (Wincon.SetConsoleTitleW (LPCWSTR
         (WCHAR_A2A.To_Pointer (New_Title_16Z'Address))));
   end Set_Console_Title;

   -----------------------
   -- Get_Output_Handle --
   -----------------------
   
   function Get_Output_Handle return Winnt.HANDLE is
      Output_Handle : constant Winnt.HANDLE :=
        Winbase.GetStdHandle (Winbase.STD_OUTPUT_HANDLE);
   begin
      if Output_Handle = Winbase.INVALID_HANDLE_VALUE then
         Raise_Last_OS_Error;
      end if;

      return Output_Handle;
   end Get_Output_Handle;

   Is_Real_Console_Output : Boolean := False;

   ---------
   -- Put --
   ---------

   procedure Put (Item : Wide_Wide_String) is
   begin
      if Is_Real_Console_Output then
         declare
            Item_16 : aliased constant UTF_16_Wide_String :=
              Encode (Item);
            Written : aliased DWORD;
         begin
            Check_OS_Error
              (Wincon.WriteConsoleW (Get_Output_Handle, Item_16'Address,
               Item_16'Length, Written'Unchecked_Access, Null_Address));
         end;
      else
         Wide_Wide_Text_IO.Put (Item);
      end if;
   end Put;

   --------------
   -- Put_Line --
   --------------

   procedure Put_Line (Item : Wide_Wide_String) is
   begin
      if Is_Real_Console_Output then
         declare
            Item_16 : aliased constant UTF_16_Wide_String :=
              Encode (Item) & Characters.Wide_Latin_1.CR &
              Characters.Wide_Latin_1.LF;
            Written : aliased DWORD;
         begin
            Check_OS_Error
              (Wincon.WriteConsoleW (Get_Output_Handle, Item_16'Address,
               Item_16'Length, Written'Unchecked_Access, Null_Address));
         end;
      else
         Wide_Wide_Text_IO.Put_Line (Item);
      end if;
   end Put_Line;

   --------------
   -- New_Line --
   --------------

   CRLF_16 : aliased constant UTF_16_Wide_String :=
     (1 => Characters.Wide_Latin_1.CR,
      2 => Characters.Wide_Latin_1.LF);

   procedure New_Line is
   begin
      if Is_Real_Console_Output then
         declare
            Written : aliased DWORD;
         begin
            Check_OS_Error
              (Wincon.WriteConsoleW (Get_Output_Handle, CRLF_16'Address,
               CRLF_16'Length, Written'Unchecked_Access, Null_Address));
         end;
      else
         Wide_Wide_Text_IO.New_Line;
      end if;
   end New_Line;

   --------------------------------
   -- Detect_Real_Console_Output --
   --------------------------------

   function Detect_Real_Console_Output return Boolean is
      Output_Handle : constant Winnt.HANDLE := Get_Output_Handle;
      Output_Type : aliased Win32.DWORD;
   begin
      Output_Type := Winbase.GetFileType (Output_Handle);
      if Output_Type = Winbase.FILE_TYPE_UNKNOWN then
         Check_OS_Error (Winbase.GetLastError);
      end if;

      Output_Type := Output_Type and not Winbase.FILE_TYPE_REMOTE;
      if Output_Type = Winbase.FILE_TYPE_CHAR then
         if 0 = Wincon.GetConsoleMode (Output_Handle,
                                       Output_Type'Unchecked_Access)
         then
            Output_Type := Winbase.GetLastError;
            if Output_Type /= Winerror.ERROR_INVALID_HANDLE then
               Raise_Last_OS_Error (Output_Type);
            end if;
         else
            return True;
         end if;
      end if;
      return False;
   end Detect_Real_Console_Output;

begin
   Is_Real_Console_Output := Detect_Real_Console_Output;
end Windows_Support.Console;
