///////////////////////////////////////////////////////////////////////////////
// ddex4_api.dpr
///////////////////////////////////////////////////////////////////////////////
// Delphi conversion of the ddex4 example contained in Microsoft's DirectX sdk.
// Based on Eric Unger's conversions of the DirectX headers. They're available
// at http://www.delphi-jedi.org. Bug reports to meyerhoff@earthling.net.
///////////////////////////////////////////////////////////////////////////////
// Description:
// Direct Draw example program 6. Adds functionality to example program 3.
// Creates a flipping surface and loads a bitmap image into an offscreen
// surface. Uses BltFast to copy portions of the offscreen surface to the back
// buffer to generate an animation. Illustrates watching return code from
// BltFast to prevent image tearing. This program requires 1.2 Meg of video
// ram.
///////////////////////////////////////////////////////////////////////////////
program ddex6;
{$APPTYPE GUI}
{$MODE DELPHI}
{$H+}
//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------
uses
  Windows,

  DirectDraw,
  SysUtils,
  DDUtil,
  DirectDrawFPC, cpu;

const
  //---------------------------------------------------------------------------
  // Local definitions
  //---------------------------------------------------------------------------
  NAME  : PChar = 'DDExample6';
  TITLE : PChar = 'Direct Draw Example 6';

var
  //---------------------------------------------------------------------------
  // Global data
  //---------------------------------------------------------------------------
  g_pDD         : IDirectDraw4;                // DirectDraw object
  g_pDDSPrimary : IDirectDrawSurface4;         // DirectDraw primary surface
  g_pDDSOne     : IDirectDrawSurface4;         // Offscreen surface 1
  g_pDDPal      : IDIRECTDRAWPALETTE;          // The primary surface palette
  g_pDDClipper  : IDirectDrawClipper;          // The clipper object
  g_bActive     : Boolean = False;             // Is application active?

  //---------------------------------------------------------------------------
  // Local data
  //---------------------------------------------------------------------------
  // Name of our bitmap resource.
  szBitmap : PChar = 'all.bmp';

  framecount : Longint = 0;

var
	z : TTimer;

//-----------------------------------------------------------------------------
// Name: ReleaseAllObjects
// Desc: Finished with all objects we use; release them
//-----------------------------------------------------------------------------
procedure ReleaseAllObjects;
begin
  if Assigned(g_pDD) then
    begin
      if Assigned(g_pDDSPrimary) then
        begin
          g_pDDSPrimary := nil;
        end;
      if Assigned(g_pDDSOne) then
        begin
          g_pDDSOne := nil;
        end;
      if Assigned(g_pDDPal) then
        begin
          g_pDDPal := nil;
        end;
      g_pDD := nil;
    end;
end;

//-----------------------------------------------------------------------------
// Name: InitFail
// Desc: This function is called if an initialization function fails
//-----------------------------------------------------------------------------
function InitFail(h_Wnd : HWND; hRet : HRESULT; Text : string) : HRESULT;
begin
  ReleaseAllObjects;
  MessageBox(h_Wnd, PChar(Text + ': ' + DDErrorString(hRet)), TITLE, MB_OK);
  DestroyWindow(h_Wnd);
  Result := hRet;
end;

//-----------------------------------------------------------------------------
// Name: RestoreAll()
// Desc: Restore all lost objects
//-----------------------------------------------------------------------------
function RestoreAll : HRESULT;
var
  hRet : HRESULT;
begin
  hRet := IDirectDrawSurface4__Restore(g_pDDSPrimary);
  if hRet = DD_OK then
    begin
      hRet := IDirectDrawSurface4__Restore(g_pDDSOne);
      if hRet = DD_OK then
        begin
          hRet := DDReLoadBitmap(g_pDDSOne, szBitmap);
          hRet := DD_OK;
        end;
    end;
  Result := hRet;
end;

//-----------------------------------------------------------------------------
// Name: UpdateFrame
// Desc: Displays the proper image for the page
//-----------------------------------------------------------------------------
var
  lastTickCount : DWORD = 0;
  currentFrame : Integer = 0;

procedure UpdateFrame(aHWnd : HWnd);
const
  delay : DWORD = 15;
var
  thisTickCount : DWORD;
  dstRect, rcRect : TRect;
  hRet : HRESULT;
  h_DC : hdc;
  rc : TRect; size :TSize;
  szMsg, temp : AnsiString;
  myDesc : TDDSurfaceDesc2;
begin
  // Decide which frame will be blitted next

  GetClientRect(aHWnd, @dstRect);
  ClientToScreen(aHWnd, lpPoint(@dstRect));
  ClientToScreen(aHWnd, lpPoint(@dstRect.right));
  // Blit the stuff for the next frame
  move(dstrect, rcRect, sizeof(rcRect));
  dec(rcRect.right, rcRect.Left); rcRect.Left := 0;
  dec(rcRect.bottom, rcRect.Top); rcRect.Top := 0;

	fillchar(myDesc, sizeof(myDesc), #0);
	myDesc.dwSize := sizeof(myDesc);
	IDirectDrawSurface4_GetSurfaceDesc(g_pDDSPrimary, myDesc);
	str(myDesc.dwWidth, temp);
	szMsg := 'Width : ' + temp;
	str(myDesc.dwHeight, temp);
	szMsg := szMsg + ' Height : ' + temp;

	repeat
      hRet := IDirectDrawSurface4_Blt(g_pDDSPrimary, @dstRect, g_pDDSOne, @rcRect, DDBLT_WAIT, nil);
      if hRet = DD_OK then
        Break;
      if hRet = DDERR_SURFACELOST then
        begin
          hRet := RestoreAll;
          if hRet <> DD_OK then
            Exit;
        end;
     until hRet <> DDERR_WASSTILLDRAWING;

  if IDirectDrawSurface4_GetDC(g_pDDSPrimary, h_DC) = DD_OK then
    begin
      SetBkColor(h_DC, RGB(0, 0, 255));
      SetTextColor(h_DC, RGB(255, 255, 0));
      GetClientRect(ahWnd, @rc);
	  DrawText(h_DC, PChar(szMsg), -1, @dstRect, DT_CENTER or DT_VCENTER or DT_SINGLELINE);
      IDirectDrawSurface4_ReleaseDC(g_pDDSPrimary, h_DC);
    end;

    inc(framecount);
end;

//-----------------------------------------------------------------------------
// Name: WindowProc
// Desc: The Main Window Procedure
//-----------------------------------------------------------------------------
function WindowProc(h_Wnd: HWND; aMSG: Cardinal; wParam: Cardinal; lParam: Integer) : Integer; stdcall;
var
  hRet : HRESULT;
  s : string;
begin
  case aMSG of
    // Pause if minimized
    WM_ACTIVATE:
      begin
      	z.stop;
        if HIWORD(wParam) = 0 then
          g_bActive := True
        else
          g_bActive := False;
        z.start;
        Result := 0;
        Exit;
      end;
    WM_CREATE:
    	begin
    		z := TZentimer.create;
    		z.start;
    		result := 0;
    		exit;
    	end;
    // Clean up and close the app
    WM_DESTROY:
      begin
      	z.stop;
      	str(z.time:5:5, s);
      	Messagebox(0, pchar('Time : ' + s), 'Performance', MB_OK);
      	str(framecount, s);
      	Messagebox(0, pchar('Framecount : ' + s), 'Performance', MB_OK);
      	str((framecount / (z.time * z.resolution)), s);
      	Messagebox(0, pchar('FPS : ' + s), 'Performance', MB_OK);
        ReleaseAllObjects;
        PostQuitMessage(0);
        Result := 0;
        Exit;
      end;
    // Handle any non-accelerated key commands
    WM_KEYDOWN:
      begin
        case wParam of
          VK_ESCAPE,
          VK_F12:
            begin
              PostMessage(h_Wnd, WM_CLOSE, 0, 0);
              Result := 0;
              Exit;
            end;
        end;
      end;
    WM_PALETTECHANGED:
      begin
        if (h_Wnd <> wParam) then
          begin
            hRet := IDirectDrawSurface4_SetPalette(g_pDDSPrimary, g_pDDPal);
            hRet := DDReLoadBitmap(g_pDDSOne, szBitmap);
            Result := 0;
          end else
            Result := 1;
      end;
    WM_QUERYNEWPALETTE:
      begin
        if (Assigned(g_pDDPal)) then
          begin
            hRet := IDirectDrawSurface4_SetPalette(g_pDDSPrimary, g_pDDPal);
            DDReLoadBitmap(g_pDDSOne, szBitmap);
          end;
        Result := 1;
      end;
    WM_PAINT:
      begin
          UpdateFrame(h_Wnd);
        Result := 0;
      end;
    end;

  Result := DefWindowProc(h_Wnd, aMSG, wParam, lParam);
end;

//-----------------------------------------------------------------------------
// Name: InitApp
// Desc: Do work required for every instance of the application:
//          Create the window, initialize data
//-----------------------------------------------------------------------------
function InitApp(hInst : THANDLE; nCmdShow : Integer) : HRESULT;
var
  h_Wnd : HWND;
  wc : WNDCLASS;
  ddsd : TDDSurfaceDesc2;
  ddscaps : TDDSCaps2;
  hRet : HRESULT;
  pddTemp : IDirectDraw;
begin
  // Set up and register window class
  wc.style := CS_HREDRAW or CS_VREDRAW;
  wc.lpfnWndProc := @WindowProc;
  wc.cbClsExtra := 0;
  wc.cbWndExtra := 0;
  wc.hInstance := hInst;
  wc.hIcon := LoadIcon(hInst, 'MAINICON');
  wc.hCursor := LoadCursor(0, IDC_ARROW);
  wc.hbrBackground := GetStockObject(GRAY_BRUSH);
  wc.lpszMenuName := nil;
  wc.lpszClassName := NAME;
  RegisterClass(wc);

  // Create a window
  h_Wnd := CreateWindowEx(WS_EX_OVERLAPPEDWINDOW,
                          NAME,
                          TITLE,
                          WS_OVERLAPPEDWINDOW,
                          CW_USEDEFAULT,
                          CW_USEDEFAULT,
                          CW_USEDEFAULT,
                          CW_USEDEFAULT,
                          0,
                          0,
                          hInst,
                          nil);
  if h_Wnd = 0 then
    begin
      Result := 0;
      Exit;
    end;

  ///////////////////////////////////////////////////////////////////////////
  // Create the main DirectDraw object
  ///////////////////////////////////////////////////////////////////////////
  hRet := DirectDrawCreate(nil, pDDTemp, nil);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'DirectDrawCreate FAILED');
      Exit;
    end;
  hRet := IDirectDraw_QueryInterface(pDDTemp, IID_IDirectDraw4, g_pDD);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'QueryInterface FAILED');
      Exit;
    end;
  pDDTemp := nil;

  // Get exclusive mode
  hRet := IDirectDraw4_SetCooperativeLevel(g_pDD, h_Wnd, DDSCL_NORMAL);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetCooperativeLevel FAILED');
      Exit;
    end;

  // Create the primary surface with 1 back buffer
  FillChar(ddsd, SizeOf(ddsd), 0);
  ddsd.dwSize := SizeOf(ddsd);
  ddsd.dwFlags := DDSD_CAPS;
  ddsd.ddsCaps.dwCaps := DDSCAPS_PRIMARYSURFACE;
  hRet := IDirectDraw4_CreateSurface(g_pDD, ddsd, g_pDDSPrimary, nil);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'CreateSurface FAILED');
      Exit;
    end;

  // Create and set the palette
  g_pDDPal := DDLoadPalette(g_pDD, szBitmap);
  if g_pDDPal = nil then
    begin
      Result := InitFail(h_Wnd, hRet, 'DDLoadPalette FAILED');
      Exit;
    end;
  // Ignore the SetPalette() function result because on RGB video modes
  // it fails
  hRet := IDirectDrawSurface4_SetPalette(g_pDDSPrimary, g_pDDPal);

  // Create and set clipper
  hRet := IDirectDraw4_CreateClipper(g_pDD, 0, g_pDDClipper, nil);
  if (hRet <> DD_OK) then
    begin
      Result := InitFail(h_Wnd, hRet, 'CreateClipper FAILED');
      Exit;
    end;
  hRet := IDirectDrawClipper_SetHWnd(g_pDDClipper, 0, h_Wnd);
  if (hRet <> DD_OK) then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetHWnd FAILED');
      Exit;
    end;
  hRet := IDirectDrawSurface4_SetClipper(g_pDDSPrimary, g_pDDClipper);
  if (hRet <> DD_OK) then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetClipper FAILED');
      Exit;
    end;

  // Create the offscreen surface, by loading our bitmap.
  g_pDDSOne := DDLoadBitmap(g_pDD, szBitmap, 0, 0);
  if g_pDDSOne = nil then
    begin
      Result := InitFail(h_Wnd, hRet, 'DDLoadBitmap FAILED');
      Exit;
    end;

  // Set the color key for this bitmap (black)
  DDSetColorKey(g_pDDSOne, RGB(0, 0, 0));

  ShowWindow(h_Wnd, nCmdShow);
  UpdateWindow(h_Wnd);
  SetFocus(h_Wnd);

  Result := DD_OK;
end;

//-----------------------------------------------------------------------------
// Name: WinMain
// Desc: Initialization, message loop
//-----------------------------------------------------------------------------
var
  aMSG : MSG;
begin
  if InitApp(GetModuleHandle(nil), SW_SHOW) <> DD_OK then
    begin
      Exit;
    end;

    while GetMessage(@aMSG, 0, 0, 0) do
      begin
        TranslateMessage(aMSG);
        DispatchMessage(aMSG);
      end;
end.
