///////////////////////////////////////////////////////////////////////////////
// ddex4_api.dpr
///////////////////////////////////////////////////////////////////////////////
// Delphi conversion of the ddex4 example contained in Microsoft's DirectX sdk.
// Based on Eric Unger's conversions of the DirectX headers. They're available
// at http://www.delphi-jedi.org. Bug reports to meyerhoff@earthling.net.
///////////////////////////////////////////////////////////////////////////////
// Description:
// Direct Draw example program 4. Adds functionality to example program 3.
// Creates a flipping surface and loads a bitmap image into an offscreen
// surface. Uses BltFast to copy portions of the offscreen surface to the back
// buffer to generate an animation. Illustrates watching return code from
// BltFast to prevent image tearing. This program requires 1.2 Meg of video
// ram.
///////////////////////////////////////////////////////////////////////////////
program ddex4_api;
{$MODE DELPHI}

//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------
uses
  Windows,

  DirectDraw,
  SysUtils,
  DDUtil,
  DirectDrawFPC;

const
  //---------------------------------------------------------------------------
  // Local definitions
  //---------------------------------------------------------------------------
  NAME  : PChar = 'DDExample4';
  TITLE : PChar = 'Direct Draw Example 4';

var
  //---------------------------------------------------------------------------
  // Global data
  //---------------------------------------------------------------------------
  g_pDD         : IDirectDraw4;                // DirectDraw object
  g_pDDSPrimary : IDirectDrawSurface4;         // DirectDraw primary surface
  g_pDDSBack    : IDirectDrawSurface4;         // DirectDraw back surface
  g_pDDSOne     : IDirectDrawSurface4;         // Offscreen surface 1
  g_pDDPal      : IDIRECTDRAWPALETTE;          // The primary surface palette
  g_bActive     : Boolean = False;             // Is application active?

  //---------------------------------------------------------------------------
  // Local data
  //---------------------------------------------------------------------------
  // Name of our bitmap resource.
  szBitmap : PChar = 'all.bmp';

//-----------------------------------------------------------------------------
// Name: ReleaseAllObjects
// Desc: Finished with all objects we use; release them
//-----------------------------------------------------------------------------
procedure ReleaseAllObjects;
begin
  if Assigned(g_pDD) then
    begin
      if Assigned(g_pDDSBack) then
        begin
          g_pDDSBack := nil;
        end;
      if Assigned(g_pDDSPrimary) then
        begin
          g_pDDSPrimary := nil;
        end;
      if Assigned(g_pDDSOne) then
        begin
          g_pDDSOne := nil;
        end;
      if Assigned(g_pDDPal) then
        begin
          g_pDDPal := nil;
        end;
      g_pDD := nil;
    end;
end;

//-----------------------------------------------------------------------------
// Name: InitFail
// Desc: This function is called if an initialization function fails
//-----------------------------------------------------------------------------
function InitFail(h_Wnd : HWND; hRet : HRESULT; Text : string) : HRESULT;
begin
  ReleaseAllObjects;
  MessageBox(h_Wnd, PChar(Text + ': ' + DDErrorString(hRet)), TITLE, MB_OK);
  DestroyWindow(h_Wnd);
  Result := hRet;
end;

//-----------------------------------------------------------------------------
// Name: RestoreAll()
// Desc: Restore all lost objects
//-----------------------------------------------------------------------------
function RestoreAll : HRESULT;
var
  hRet : HRESULT;
begin
  hRet := IDirectDrawSurface4__Restore(g_pDDSPrimary);
  if hRet = DD_OK then
    begin
      hRet := IDirectDrawSurface4__Restore(g_pDDSOne);
      if hRet = DD_OK then
        begin
          hRet := DDReLoadBitmap(g_pDDSOne, szBitmap);
          hRet := DD_OK;
        end;
    end;
  Result := hRet;
end;

//-----------------------------------------------------------------------------
// Name: UpdateFrame
// Desc: Displays the proper image for the page
//-----------------------------------------------------------------------------
var
  lastTickCount : array[0..2] of DWORD =  (0, 0, 0);
  currentFrame : array[0..2] of Integer = (0, 0, 0);

procedure UpdateFrame;
const
  delay : array[0..2] of DWORD = (50, 78, 13);
  xpos : array[0..2] of Integer = (288, 190, 416);
  ypos : array[0..2] of Integer = (128, 300, 256);
var
  i : Integer;
  thisTickCount : DWORD;
  rcRect : TRect;
  hRet : HRESULT;
begin
  // Decide which frame will be blitted next
  thisTickCount := GetTickCount;
  for i := 0 to 2 do
    begin
      if (thisTickCount - lastTickCount[i]) > delay[i] then
        begin
          // Move to next frame;
          lastTickCount[i] := thisTickCount;
          currentFrame[i] := currentFrame[i] + 1;
          if currentFrame[i] > 59 then
            currentFrame[i] := 0;
        end;
    end;

  // Blit the stuff for the next frame
  rcRect.left := 0;
  rcRect.top := 0;
  rcRect.right := 640;
  rcRect.bottom := 480;
  while True do
    begin
      hRet := IDirectDrawSurface4_BltFast(g_pDDSBack, 0, 0, g_pDDSOne, @rcRect, DDBLTFAST_NOCOLORKEY);
      if hRet = DD_OK then
        Break;
      if hRet = DDERR_SURFACELOST then
        begin
          hRet := RestoreAll;
          if hRet <> DD_OK then
            Exit;
        end;
      if hRet <> DDERR_WASSTILLDRAWING then
        Exit;
    end;

  for i := 0 to 2 do
    begin
      rcRect.left := currentFrame[i] mod 10 * 64;
      rcRect.top := currentFrame[i] div 10 * 64 + 480;
      rcRect.right := currentFrame[i] mod 10 * 64 + 64;
      rcRect.bottom := currentFrame[i] div 10 * 64 + 64 + 480;

      while True do
        begin
          hRet := IDirectDrawSurface4_BltFast(g_pDDSBack, xpos[i], ypos[i], g_pDDSOne, @rcRect, DDBLTFAST_SRCCOLORKEY);
          if hRet = DD_OK then
            Break;
          if hRet = DDERR_SURFACELOST then
            begin
              hRet := RestoreAll;
              if hRet <> DD_OK then
                Exit;
            end;
          if hRet <> DDERR_WASSTILLDRAWING then
            Exit;
        end;
    end;

  // Flip the surfaces
  while True do
    begin
      hRet := IDirectDrawSurface4_Flip(g_pDDSPrimary, nil, 0);
      if hRet = DD_OK then
        Break;
      if hRet = DDERR_SURFACELOST then
        begin
          hRet := RestoreAll;
          if hRet <> DD_OK then
            Break;
        end;
      if hRet <> DDERR_WASSTILLDRAWING then
        Break;
    end;
end;

//-----------------------------------------------------------------------------
// Name: WindowProc
// Desc: The Main Window Procedure
//-----------------------------------------------------------------------------
function WindowProc(h_Wnd: HWND; aMSG: Cardinal; wParam: Cardinal; lParam: Integer) : Integer; stdcall;
begin
  case aMSG of
    // Pause if minimized
    WM_ACTIVATE:
      begin
        if HIWORD(wParam) = 0 then
          g_bActive := True
        else
          g_bActive := False;
        Result := 0;
        Exit;
      end;
    // Clean up and close the app
    WM_DESTROY:
      begin
        ReleaseAllObjects;
        PostQuitMessage(0);
        Result := 0;
        Exit;
      end;
    // Handle any non-accelerated key commands
    WM_KEYDOWN:
      begin
        case wParam of
          VK_ESCAPE,
          VK_F12:
            begin
              PostMessage(h_Wnd, WM_CLOSE, 0, 0);
              Result := 0;
              Exit;
            end;
        end;
      end;
    // Turn off the cursor since this is a full-screen app
    WM_SETCURSOR:
      begin
        SetCursor(0);
        Result := 1;
        Exit;
      end;
    end;

  Result := DefWindowProc(h_Wnd, aMSG, wParam, lParam);
end;

//-----------------------------------------------------------------------------
// Name: InitApp
// Desc: Do work required for every instance of the application:
//          Create the window, initialize data
//-----------------------------------------------------------------------------
function InitApp(hInst : THANDLE; nCmdShow : Integer) : HRESULT;
var
  h_Wnd : HWND;
  wc : WNDCLASS;
  ddsd : TDDSurfaceDesc2;
  ddscaps : TDDSCaps2;
  hRet : HRESULT;
  pddTemp : IDirectDraw;
begin
  // Set up and register window class
  wc.style := CS_HREDRAW or CS_VREDRAW;
  wc.lpfnWndProc := @WindowProc;
  wc.cbClsExtra := 0;
  wc.cbWndExtra := 0;
  wc.hInstance := hInst;
  wc.hIcon := LoadIcon(hInst, 'MAINICON');
  wc.hCursor := LoadCursor(0, IDC_ARROW);
  wc.hbrBackground := GetStockObject(BLACK_BRUSH);
  wc.lpszMenuName := NAME;
  wc.lpszClassName := NAME;
  RegisterClass(wc);

  // Create a window
  h_Wnd := CreateWindowEx(WS_EX_TOPMOST,
                          NAME,
                          TITLE,
                          WS_POPUP,
                          0,
                          0,
                          GetSystemMetrics(SM_CXSCREEN),
                          GetSystemMetrics(SM_CYSCREEN),
                          0,
                          0,
                          hInst,
                          nil);

  if h_Wnd = 0 then
    begin
      Result := 0;
      Exit;
    end;

  ShowWindow(h_Wnd, nCmdShow);
  UpdateWindow(h_Wnd);
  SetFocus(h_Wnd);

  ///////////////////////////////////////////////////////////////////////////
  // Create the main DirectDraw object
  ///////////////////////////////////////////////////////////////////////////
  hRet := DirectDrawCreate(nil, pDDTemp, nil);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'DirectDrawCreate FAILED');
      Exit;
    end;
  hRet := IDirectDraw_QueryInterface(pDDTemp, IID_IDirectDraw4, g_pDD);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'QueryInterface FAILED');
      Exit;
    end;
  pDDTemp := nil;

  // Get exclusive mode
  hRet := IDirectDraw4_SetCooperativeLevel(g_pDD, h_Wnd, DDSCL_EXCLUSIVE or DDSCL_FULLSCREEN);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetCooperativeLevel FAILED');
      Exit;
    end;

  // Set the video mode to 640x480x8
  hRet := IDirectDraw4_SetDisplayMode(g_pDD, 640, 480, 8, 0, 0);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetDisplayMode FAILED');
      Exit;
    end;

  // Create the primary surface with 1 back buffer
  FillChar(ddsd, SizeOf(ddsd), 0);
  ddsd.dwSize := SizeOf(ddsd);
  ddsd.dwFlags := DDSD_CAPS or DDSD_BACKBUFFERCOUNT;
  ddsd.ddsCaps.dwCaps := DDSCAPS_PRIMARYSURFACE or DDSCAPS_FLIP or DDSCAPS_COMPLEX;
  ddsd.dwBackBufferCount := 1;
  hRet := IDirectDraw4_CreateSurface(g_pDD, ddsd, g_pDDSPrimary, nil);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'CreateSurface FAILED');
      Exit;
    end;

  // Get a pointer to the back buffer
  FillChar(ddscaps, SizeOf(ddscaps), 0);
  ddscaps.dwCaps := DDSCAPS_BACKBUFFER;
  hRet := IDirectDrawSurface4_GetAttachedSurface(g_pDDSPrimary, ddscaps, g_pDDSBack);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'GetAttachedSurface FAILED');
      Exit;
    end;

  // Create and set the palette
  g_pDDPal := DDLoadPalette(g_pDD, szBitmap);
  if g_pDDPal = nil then
    begin
      Result := InitFail(h_Wnd, hRet, 'DDLoadPalette FAILED');
      Exit;
    end;
  hRet := IDirectDrawSurface4_SetPalette(g_pDDSPrimary, g_pDDPal);
  if hRet <> DD_OK then
    begin
      Result := InitFail(h_Wnd, hRet, 'SetPalette FAILED');
      Exit;
    end;

  // Create the offscreen surface, by loading our bitmap.
  g_pDDSOne := DDLoadBitmap(g_pDD, szBitmap, 0, 0);
  if g_pDDSOne = nil then
    begin
      Result := InitFail(h_Wnd, hRet, 'DDLoadBitmap FAILED');
      Exit;
    end;

  // Set the color key for this bitmap (black)
  DDSetColorKey(g_pDDSOne, RGB(0, 0, 0));

  Result := DD_OK;
end;

//-----------------------------------------------------------------------------
// Name: WinMain
// Desc: Initialization, message loop
//-----------------------------------------------------------------------------
var
  aMSG : MSG;
begin
  if InitApp(GetModuleHandle(nil), SW_SHOW) <> DD_OK then
    begin
      Exit;
    end;

  while True do
    begin
      if PeekMessage(@aMsg, 0, 0, 0, PM_NOREMOVE) then
        begin
          if not GetMessage(@aMsg, 0, 0, 0) then
            Exit;
          TranslateMessage(aMsg);
          DispatchMessage(aMsg);
        end
      else if g_bActive then
        begin
            UpdateFrame;
        end
      else
        begin
          // Make sure we go to sleep if we have nothing else to do
          WaitMessage;
        end;
    end;
end.
